/*=========================================================================
 *=========================================================================
 == [DISCLAIMER]: THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS   ==
 == RELEASED "AS IS".  THE U.S. GOVERNMENT MAKES NO WARRANTY OF ANY     ==
 == KIND, EXPRESS OR IMPLIED, CONCERNING THIS SOFTWARE AND ANY          ==
 == ACCOMPANYING DOCUMENTATION, INCLUDING, WITHOUT LIMITATION, ANY      ==
 == WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  ==
 == IN NO EVENT WILL THE U.S. GOVERNMENT BE LIABLE FOR ANY DAMAGES      ==
 == ARISING OUT OF THE USE, OR INABILITY TO USE, THIS SOFTWARE OR ANY   ==
 == ACCOMPANYING DOCUMENTATION, EVEN IF INFORMED IN ADVANCE OF THE      ==
 == POSSIBLITY OF SUCH DAMAGES.                                         ==
 =========================================================================
 =========================================================================*/

/*************************************************************************
 *                                                                       *
 *                      Routine:  convert884.c                           *
 *                       Author:  J. Querns, Veda Inc.                   *
 *                                5200 Springfield Pike, Suite 200       *
 *                                Dayton, OH 45431                       *
 *                                (937) 255-1116, Ext 2818               *
 *                                                                       *
 *                         Date:  3 June 1997                            *
 *                                                                       *
 *  Description:  This routine reads an MIT/LL ADTS (ATRWG) 8-8-4 for-   *
 *                matted file, extracts the specified polarization sub-  *
 *                image, and writes it out to a RAW (header-less) file   *
 *                in the data output format specified.  The command      *
 *                syntax is:                                             *
 *                                                                       *
 *                convert884 <infile>           {REQUIRED}               *
 *                           <outfile>          {REQUIRED}               *
 *                                                                       *
 *                     -h        {Additionally dump headers...}          *
 *                                                                       *
 *                         If this switch is selected, the program will  *
 *                         ADDITIONALLY dump the MISSION (header-2) info *
 *                         that follows the HH image header and the      *
 *                         specific polarization image header (header-1) *
 *                         to a file designated as: <outfile>.hdr .  The *
 *                         default is to NOT WRITE out the headers..     *
 *                                                                       *
 *                     -p<popt>  {Polarization option}                   *
 *                                                                       *
 *                         0: HH polarization {def}                      *
 *                         1: HV polarization                            *
 *                         2: VH polarization                            *
 *                         3: VV polarization                            *
 *                                                                       *
 *                     -d<dopt>  {Output DATATYPE option}                *
 *                                                                       *
 *                         0: RAW Complex Image {def}                    *
 *                         1: RAW Float MAGNITUDE Image                  *
 *                         2: RAW Float MAGNITUDE**2 Image (SIGMA-0)     *
 *                         3: RAW Float RCS Image                        *
 *                                                                       *
 *               If no option switches are entered, the program will use *
 *               the default options (HH polarization, Complex image).   *
 *                                                                       *
 *  <ADTS FILE FORMAT>:                                                  *
 *                                                                       *
 *  For information on the ADTS 8-8-4 File format, read the file,        *
 *  "ADTS_README"...                                                     *
 *                                                                       *
 *                                                                       *
 *  <CONVERT884 NOTES>:                                                  *
 *                                                                       *
 *  This routine will perform the following steps:                       *
 *                                                                       *
 *    a) Process command line arguments.                                 *
 *                                                                       *
 *    b) Call the subroutine getImgLoc(), which will return the starting *
 *       locations (i.e. file offsets) for each of the included          *
 *       polarization image subfiles and headers.                        *
 *                                                                       *
 *    c) Read, convert, and write out data based on the output options   *
 *       selected...                                                     *
 *                                                                       *
 *  <GENERAL ASSUMPTIONS>:                                               *
 *                                                                       *
 *    a) Every image sub-frame should have the same number of samples    *
 *       per lines and the same number of lines per image. These are     *
 *       retrieved ONCE from the first sub-image in the file.            *
 *                                                                       *
 *    b) What may be different is the number of records per image and    *
 *       the number of bytes per record. These must be extracted for     *
 *       each sub-frame first.  If there is only one image per file, the *
 *       code will detect it and drop to the next step.                  *
 *                                                                       *
 *       This routine should read single polarization image ADTS files   *
 *       or target "chips".                                              *
 *                                                                       *
 *  <PROCESSING or CONVERSIONS>:                                         *
 *                                                                       *
 *    a) COMPLEX data is created using the 8-8-4 extraction formula      *
 *       mentioned in the file: "ADTS_README".                           *
 *       file...                                                         *
 *                                                                       *
 *       NOTE: The extracted complex data pixels are UNITLESS values     *
 *             equal to the SQRT(Backscatter-Coefficient).  The "back-   *
 *             scattering coeficient" (or SIGMA-0) is defined as:        *
 *                                                                       *
 *              (RCS of a Radar Resolution Cell, SqMeters) /             *
 *              (Ground Area of Radar Resolution Cell, SqMeters)         *
 *                                                                       *
 *             To obtain the backscattering coefficient value, we just   *
 *             multiply the extracted pixel value by its complex con-    *
 *             jugate value (equivalent to MAGNITUDE**2 image)           *
 *                                                                       *
 *                SIGMA-0 = (a + bi) * (a - bi) = (a**2 + b**2)          *
 *                                                                       *
 *             To obtain the RCS value, we first multiply the complex    *
 *             pixel value by a "sigma_0_to_rcs" conversion factor ex-   *
 *             tracted from the image header, scale it, and then multiply*
 *             it by its complex conjugate value.                        *
 *                                                                       *
 *    b) MAGNITUDE:   Sqrt (Re**2 + Im**2)                               *
 *                                                                       *
 * Calls: getImgLoc() -- This subroutine returns the file location of    *
 *                       the specified polarization sub-image, the sub-  *
 *                       image header-1, the mission header (header-2),  *
 *                       and the number of samples and lines...          *
 *                                                                       *
 *        dumpHdr()  --  This subroutine will dump the polarization-     *
 *                       specific sub-image header (header-1) as well as *
 *                       the MISSION header (header-2)                   *
 *                                                                       *
 *        read_switch() -- Extract command line arguments...             *
 *                                                                       *
 *************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "adts.h"

#define  SQUARED(a)   ( (a) * (a) )

#define FLAG_ON  1
#define FLAG_OFF 0

extern int getImgLoc ();	/* Extracts img & hdr locations   */
extern int dumpHdr ();		/* Dumps image headers...         */
extern char *read_switch ();	/* ExtrACTS ARGUMENTS...          */

main (argc, argv)

     int argc;
     char *argv[];

{
  FILE *OUTfp;			/* Ptr to output OUT image...     */
  FILE *ADTSfp;			/* Ptr to ADTS file....           */
  char *ADTSname;		/* ADTS filename...               */
  char *OUTname;		/* OUT filename...                */
  char *HDR2;			/* Ptr to potential header2 array */

  char *HDR0ext = ".hdr0";
  char *HDR1ext = ".hdr1";
  char *HDR2ext = ".hdr2";
  char *HDR3ext = ".hdr3";

  char HDRname[80];		/* Array to hold header file name */
  float *frec;			/* Ptr to array of float num..    */
  struct COMPLEXN *crec;	/* Ptr to array of complex num..  */
  struct ADTSPIX *adtsrec;	/* Ptr to array of ADTS pixels... */
  struct ADTSImgOff *Offptr;	/* Ptr to sub-image offset struct */

  long i, l, s, bpl;
  long ns, nl;			/* Num of lines and samples in img */
  long hdr2len, hdr2loc;
  char *phdr2;
  int retv;			/* Return value (i.e. status)..   */
  int poltype;			/* Polarization image type wanted */
  int datatype;			/* Output data type...            */
  int HDRflag;			/* Def: don't also dump header    */
  float sigma0_to_rcs;		/* Conversion factor...           */
  float f1, f2;			/* Temp float variables...        */
  double d1, d2;		/* Temp double float variables..  */

  char *HDRopt;			/* Ptr to header opt selection..  */

  char *POLopt;			/* Ptr to variable pol option     */
  char *POLopt0;		/* Ptr to HH pol selection        */
  char *POLopt1;		/* Ptr to HV pol selection        */
  char *POLopt2;		/* Ptr to VH pol selection        */
  char *POLopt3;		/* Ptr to VV pol selection        */

  char *DATAopt;		/* Ptr to variable datatype option */
  char *DATAopt0;		/* Ptr to COMPLEX datatype opt    */
  char *DATAopt1;		/* Ptr to MAG datatype output opt */
  char *DATAopt2;		/* Ptr to MAG-SQ datatype output opt */
  char *DATAopt3;		/* Ptr to RCS datatype output opt */

  /* Check for correct arguments */

  if (argc < 2)
    {
      fprintf (stderr,
	       "\n\nUsage: %s -i <infile> -o <outfile> -h -p <popt> -d <dopt>\n", argv[0]);
      fprintf (stderr,
	       "\n(ADTS Header ): [-h] Dump image headers to <outfile>.hdr");
      fprintf (stderr,
      "\n(Polarization): [-p] <popt> --> 0: HH [def], 1: HV, 2: VH, 3: VV");
      fprintf (stderr,
	       "\n(Output Type) : [-d] <dopt> --> <dopt>:");
      fprintf (stderr,
	       "\n                     0: RAW Complex Image [default]");
      fprintf (stderr,
	       "\n                     1: RAW MAGNITUDE Image");
      fprintf (stderr,
	       "\n                     2: RAW MAGNITUDE**2 [SIGMA-0] Image");
      fprintf (stderr,
	       "\n                     3: RAW RCS Image\n\n");
      exit (1);
    }

  /* Extract Optional Switch Arguments */

  ADTSname = read_switch (&argc, argv, "-i", 1, NULL);
  OUTname = read_switch (&argc, argv, "-o", 1, NULL);
  HDRopt = read_switch (&argc, argv, "-h", 0, NULL);

/****
 This extracts POL options in one of two formats:
 "-p <popt>" or "-p<popt>". Note that the 1st format
 is has a space between the switch and the argument.
 The 2nd format does not.  The 2nd format was added
 to allow the user to select the format desired without
 worrying about fumbling with the keys and forgetting
 to enter a space between the switch and the argument,
 as you must do with the 1st format...
 ****/
  POLopt = read_switch (&argc, argv, "-p", 1, NULL);
  POLopt0 = read_switch (&argc, argv, "-p0", 0, NULL);
  POLopt1 = read_switch (&argc, argv, "-p1", 0, NULL);
  POLopt2 = read_switch (&argc, argv, "-p2", 0, NULL);
  POLopt3 = read_switch (&argc, argv, "-p3", 0, NULL);

/****
 This extracts DATA OUTPUT options in one of two formats:
 "-d <dopt>" or "-d<dopt>". Note that the 1st format
 is has a space between the switch and the argument.
 The 2nd format does not.  The 2nd format was added
 to allow the user to select the format desired without
 worrying about fumbling with the keys and forgetting
 to enter a space between the switch and the argument,
 as you must do with the 1st format...
 ****/
  DATAopt = read_switch (&argc, argv, "-d", 1, NULL);
  DATAopt0 = read_switch (&argc, argv, "-d0", 0, NULL);
  DATAopt1 = read_switch (&argc, argv, "-d1", 0, NULL);
  DATAopt2 = read_switch (&argc, argv, "-d2", 0, NULL);
  DATAopt3 = read_switch (&argc, argv, "-d3", 0, NULL);

/****************************************** 
 * Check for ADTS (8-8-4) Input file name *
 ******************************************/

  if (ADTSname == (char *) NULL)
    {
      fprintf (stderr, "\n\nError: no input file entered or incorrect syntax!");
      fprintf (stderr, "\nInput file syntax = -i <infile>\n\n");
      exit (1);
    }
  else
    {
      printf ("\n\nInput  file: [%s]", ADTSname);
    }

/******************************
 * Check for Output file name *
 ******************************/

  if (OUTname == (char *) NULL)
    {
      fprintf (stderr, "\n\nError: no output file entered or incorrect syntax!");
      fprintf (stderr, "\nOutput file syntax = -o <outfile>\n\n");
      exit (1);
    }
  else
    {
      printf ("\nOutput file: [%s]", OUTname);
    }

/******************************************************** 
 * Check for header option ...                          *
 * NOTE: If 'HDRopt' = NULL, then "-h" was not inputted *
 ********************************************************/

  if (HDRopt == (char *) NULL)
    {
      HDRflag = FLAG_OFF;
    }
  else
    {
      printf ("\nDump headers selected...");
      HDRflag = FLAG_ON;
    }

/**************************************
 * Check for polarization type option *
 **************************************/

/* Check for POLopt format-2 ... */
  if (POLopt0 != (char *) NULL)
    {
      printf ("\nADTS [HH] polarization selected...");
      poltype = 0;		/* HH selected */
    }
  else if (POLopt1 != (char *) NULL)
    {
      printf ("\nADTS [HV] polarization selected...");
      poltype = 1;		/* HV selected */
    }
  else if (POLopt2 != (char *) NULL)
    {
      printf ("\nADTS [VH] polarization selected...");
      poltype = 2;		/* VH selected */
    }
  else if (POLopt3 != (char *) NULL)
    {
      printf ("\nADTS [VV] polarization selected...");
      poltype = 3;		/* VV selected */
    }
  else if ((POLopt == (char *) NULL) &&
	   (POLopt0 == (char *) NULL) &&
	   (POLopt1 == (char *) NULL) &&
	   (POLopt2 == (char *) NULL) &&
	   (POLopt3 == (char *) NULL))
    {
      printf ("\nSpecific POL not selected..using HH..");
      poltype = 0;
    }
  else
    {
      poltype = atoi (POLopt);
      if ((poltype < 0) || (poltype > 3))
	{
	  fprintf (stderr, "\n\nError: invalid polarization type!");
	  fprintf (stderr, "\n  -p <popt> ");
	  fprintf (stderr, "\n     0: HH Sub-Image");
	  fprintf (stderr, "\n     1: HV Sub-Image");
	  fprintf (stderr, "\n     2: VH Sub-Image");
	  fprintf (stderr, "\n     3: VV Sub-Image\n\n");
	  exit (1);
	}

      switch (poltype)
	{
	case 0:		/* HH */
	  printf ("\nADTS [HH] polarization selected...");
	  break;
	case 1:		/* HV */
	  printf ("\nADTS [HV] polarization selected...");
	  break;
	case 2:		/* VH */
	  printf ("\nADTS [VH] polarization selected...");
	  break;
	case 3:		/* VV */
	  printf ("\nADTS [VV] polarization selected...");
	  break;
	}

    }

/************************************ 
 * Check for datatype output option *
 ************************************/

/* Check for DATAopt format-2 ... */
  if (DATAopt0 != (char *) NULL)
    {
      printf ("\nConversion to RAW COMPLEX selected...");
      datatype = 0;		/* COMPLEX selected */
    }
  else if (DATAopt1 != (char *) NULL)
    {
      printf ("\nConversion to RAW MAGNITUDE selected...");
      datatype = 1;		/* MAGNITUDE selected */
    }
  else if (DATAopt2 != (char *) NULL)
    {
      printf ("\nConversion to RAW MAGNITUDE-SQUARED selected...");
      datatype = 2;		/* MAGNITUDE-SQ selected */
    }
  else if (DATAopt3 != (char *) NULL)
    {
      printf ("\nConversion to RAW RCS selected...");
      datatype = 3;		/* RCS selected */
    }
  else if ((DATAopt == (char *) NULL) &&
	   (DATAopt0 == (char *) NULL) &&
	   (DATAopt1 == (char *) NULL) &&
	   (DATAopt2 == (char *) NULL) &&
	   (DATAopt3 == (char *) NULL))
    {

      printf ("\nSpecific DATATYPE not selected..using COMPLEX..");
      datatype = 0;
    }
  else
    {
      datatype = atoi (DATAopt);
      if ((datatype < 0) || (datatype > 3))
	{
	  fprintf (stderr, "\n\nError: invalid data output type!");
	  fprintf (stderr, "\n  -d <dopt> ");
	  fprintf (stderr, "\n     0: RAW Complex Image");
	  fprintf (stderr, "\n     1: RAW Magnitude Image");
	  fprintf (stderr, "\n     2: RAW Magnitude**2 Image");
	  fprintf (stderr, "\n     3: RAW RCS Image\n\n");
	  exit (1);
	}

      switch (datatype)
	{
	case 0:		/* COMPLEX */
	  printf ("\nConversion to RAW COMPLEX selected...");
	  break;
	case 1:		/* MAGNITUDE */
	  printf ("\nConversion to RAW MAGNITUDE selected...");
	  break;
	case 2:		/* MAGNITUDE-SQUARED */
	  printf ("\nConversion to RAW MAGNITUDE-SQ selected...");
	  break;
	case 3:		/* RCS */
	  printf ("\nConversion to RAW RCS selected...");
	  break;
	}

    }

  /* Open ADTS file for Reading */

  ADTSfp = fopen (ADTSname, "rb");
  if (ADTSfp == NULL)
    {
      fprintf (stderr, "\n\nError: Unable to open file: [%s]", ADTSname);
      fprintf (stderr, "\nExiting [convert884] program...\n\n");
      exit (1);			/* FAILURE */
    }

/****************************************************************************
 *        Get Sub-Image Offset Location Data for all 4 Polarizations         *
 ****************************************************************************/

  if ((retv = getImgLoc (ADTSfp)) != 0)
    {
      fprintf (stderr, "\n\nError: in obtaining offsetlocation data");
      fprintf (stderr, "\nExiting [convert884] program...\n\n");
      fclose (ADTSfp);
      exit (1);			/* FAILURE */
    }

/* 
   Check for existence of a header-2..if no header-2 and RCS option
   selected, output error as no sigma_0 to RCS conversion factor would
   be available..
 */

  if ((datatype == 3) &&
      (hhOff.hdr2len == 0))
    {
      fprintf (stderr,
	       "\n\nUnable to extract SIGMA_0 to RCS conversion factor ");
      fprintf (stderr,
	       "\nfrom which to create valid RCS image...");
      fprintf (stderr, "\nExiting [convert884] program...\n\n");
      fclose (ADTSfp);
      exit (1);			/* FAILURE */
    }
  else
    {
      ns = hhOff.nsamps;
      nl = hhOff.nlines;
    }


  printf ("\n\nNumber of samples per image line = [%ld]", ns);
  printf ("\nNumber of lines per image        = [%ld]", nl);

/****************************************************************************
 *                   RETRIEVE, PROCESS, & WRITE OUT DATA                    *
 *                       DUMP HEADERS, IF NECESSARY                         *
 ****************************************************************************/

/*********************************
 * Allocate array memories...    *
 *********************************/

  adtsrec = (struct ADTSPIX *) malloc (ns * sizeof (struct ADTSPIX));

  if (adtsrec == NULL)
    {
      fprintf (stderr,
	       "\n\nError[%s]: Unable to allocate ADTS record memory!",
	       argv[0]);
      fprintf (stderr, "\nExiting [convert884] program...\n\n");
      fclose (ADTSfp);		/* Close input file */
      exit (1);
    }

  crec = (struct COMPLEXN *) malloc (ns * sizeof (struct COMPLEXN));

  if (crec == NULL)
    {
      fprintf (stderr,
	       "\n\nError[%s]: Unable to allocate complex number memory!",
	       argv[0]);
      fprintf (stderr, "\nExiting [convert884] program...\n\n");
      free (adtsrec);		/* Free ADTS memory      */
      fclose (ADTSfp);		/* Close input file */
      exit (1);
    }

  if (datatype != 0)		/* If TRUE, also allocate float array */
    {
      frec = (float *) malloc (ns * sizeof (float));

      if (frec == NULL)
	{
	  fprintf (stderr,
	  "\n\nError[%s]: Unable to allocate float number memory!", argv[0]);
	  fprintf (stderr, "\nExiting [convert884] program...\n\n");
	  free (adtsrec);	/* Free ADTS memory */
	  free (crec);		/* Free complex memory */
	  fclose (ADTSfp);	/* Close input file */
	  exit (1);
	}
    }

/***************************************
 * Get sigma0_to_rcs conversion factor *
 * if datatype = 3 (RCS option)...     *
 ***************************************/

  if (datatype == 3)
    {
      if (hhOff.hdr2len > 0)	/* i.e. a header-2 exists */
	{
	  hdr2loc = hhOff.hdr2;
	  hdr2len = hhOff.hdr2len;
	}
      else
	{
	  fprintf (stderr,
		   "\n\nError[%s]: Unable to create RCS image!", argv[0]);
	  fprintf (stderr,
		"\nNo Header-2 from which to retrieve conversion factor..");
	  fprintf (stderr,
		   "\nExiting [convert884] program...\n\n");
	  free (adtsrec);	/* Free ADTS memory */
	  free (crec);		/* Free complex memory */
	  if (datatype != 0)
	    {
	      free (frec);	/* Free float memory */
	    }
	  fclose (ADTSfp);	/* Close input file */
	  exit (1);
	}

      HDR2 = (char *) malloc (hdr2len * sizeof (char));

      if (HDR2 == (char *) NULL)
	{
	  fprintf (stderr,
	       "\n\nError[%s]: Unable to allocate header memory!", argv[0]);
	  fprintf (stderr, "\nExiting [convert884] program...\n\n");
	  free (adtsrec);	/* Free ADTS memory */
	  free (crec);		/* Free complex memory */
	  if (datatype != 0)
	    {
	      free (frec);	/* Free float memory */
	    }
	  fclose (ADTSfp);	/* Close input file */
	  exit (1);
	}

      fseek (ADTSfp, hdr2loc, 0);	/* Seek to start of header 2 */
      fread (HDR2, sizeof (char), hdr2len, ADTSfp);

      phdr2 = strstr (HDR2, "SIGMA_0_TO_RCS      ");
      if (phdr2 == NULL)
	{
	  fprintf (stderr,
		   "\n\nError[%s]: Problems extracting RCS conversion factor!", argv[0]);
	  fprintf (stderr, "\nExiting [convert884] program...\n\n");
	  free (adtsrec);	/* Free ADTS memory */
	  free (crec);		/* Free complex memory */
	  if (datatype != 0)
	    {
	      free (frec);	/* Free float memory   */
	    }
	  free (HDR2);		/* Free header2 memory */
	  fclose (ADTSfp);	/* Close input file    */
	  exit (1);
	}
      phdr2 += 34;		/* Move Conversion ptr to start of data.. */
      sscanf (phdr2, "%f", &sigma0_to_rcs);

      printf ("\nSIGMA_0 to RCS conversion factor = [%f]", sigma0_to_rcs);

      free (HDR2);		/* Free header2 memory */

    }				/* End of RCS conversion factor extraction */


/***************************************
 * Open output image file for writing  *
 ***************************************/

  OUTfp = fopen (OUTname, "wb");
  if (OUTfp == NULL)
    {
      fprintf (stderr, "\n\nError: Unable to create file: [%s]", OUTname);
      fprintf (stderr, "\nExiting [convert884] program...\n\n");
      free (adtsrec);		/* Free memory       */
      free (crec);		/* Free memory       */
      fclose (ADTSfp);		/* Close input file  */
      exit (1);
    }


/*********************************
 *   Process Data....            *
 *********************************
 * Each fseek in 1st switch seeks*
 * to start of next ADTS record. *
 *********************************/

  switch (poltype)
    {
    case 0:			/* HH */
      printf ("\nProcessing ADTS 8-8-4 [HH] Sub-image...");
      break;
    case 1:			/* HV */
      printf ("\nProcessing ADTS 8-8-4 [HV] Sub-image...");
      break;
    case 2:			/* VH */
      printf ("\nProcessing ADTS 8-8-4 [VH] Sub-image...");
      break;
    case 3:			/* VV */
      printf ("\nProcessing ADTS 8-8-4 [VV] Sub-image...");
      break;
    }

  rewind (ADTSfp);		/* Rewind ADTS image file...          */
  bpl = (ns + 4) * 3;		/* Bytes per line (inc. hdr&tr pixels */

  for (l = 0; l < nl; l++)
    {
      switch (poltype)
	{
	case 0:		/* HH */
	  fseek (ADTSfp, (hhOff.img) + (l * bpl), 0);
	  break;
	case 1:		/* HV */
	  fseek (ADTSfp, (hvOff.img) + (l * bpl), 0);
	  break;
	case 2:		/* VH */
	  fseek (ADTSfp, (vhOff.img) + (l * bpl), 0);
	  break;
	case 3:		/* VV */
	  fseek (ADTSfp, (vvOff.img) + (l * bpl), 0);
	  break;
	}

      /* Skip over 1st three header pixels..   */
      fseek (ADTSfp, 9L, 1);
      /* Read one ADTS image record...         */
      fread (adtsrec, sizeof (struct ADTSPIX), ns, ADTSfp);

      for (s = 0; s < ns; s++)
	{
	  crec[s].re = (float) ((long) adtsrec[s].i << (adtsrec[s].e & 017)) / 4096.0;
	  crec[s].im = (float) ((long) adtsrec[s].q << (adtsrec[s].e & 017)) / 4096.0;
	}

      switch (datatype)
	{
	case 0:		/* (Def) RAW Complex Image */
	  fwrite (crec, sizeof (struct COMPLEXN), ns, OUTfp);

	  break;
	case 1:		/* RAW Magnitude Image */
	  for (s = 0; s < ns; s++)
	    {
	      d1 = (double) SQUARED (crec[s].re);
	      d2 = (double) SQUARED (crec[s].im);
	      frec[s] = (float) sqrt (d1 + d2);
	    }
	  fwrite (frec, sizeof (float), ns, OUTfp);

	  break;
	case 2:		/* RAW Magnitude-Squared Image */
	  for (s = 0; s < ns; s++)
	    {
	      d1 = (double) SQUARED (crec[s].re);
	      d2 = (double) SQUARED (crec[s].im);
	      frec[s] = (float) (d1 + d2);
	    }
	  fwrite (frec, sizeof (float), ns, OUTfp);

	  break;

	case 3:		/* RAW RCS Image */
	  for (s = 0; s < ns; s++)
	    {
	      d1 = (double) SQUARED ((crec[s].re * sigma0_to_rcs));
	      d2 = (double) SQUARED ((crec[s].im * sigma0_to_rcs));
	      frec[s] = (float) (d1 + d2);
	    }
	  fwrite (frec, sizeof (float), ns, OUTfp);

	  break;

	}			/* End of output datatype switch */

    }				/* End of lines loop   */

  free (adtsrec);		/* Free ADTS image record memory... */
  free (crec);			/* Free Complex image memory...     */
  if (datatype != 0)
    {
      free (frec);		/* Free float memory if not default */
    }

/****************************************************
 * Check for ADTS Header Dump; do so, if necessary..*
 ****************************************************/

  if (HDRflag != FLAG_OFF)
    {
      rewind (ADTSfp);		/* Rewind ADTS image file...          */
      strcpy (HDRname, OUTname);
      switch (poltype)
	{
	case 0:		/* HH */
	  strcat (HDRname, HDR0ext);
	  Offptr = &hhOff;
	  printf ("\nDumping [HH] headers to [%s]", HDRname);
	  break;
	case 1:		/* HV */
	  strcat (HDRname, HDR1ext);
	  Offptr = &hvOff;
	  printf ("\nDumping [HV] headers to [%s]", HDRname);
	  break;
	case 2:		/* VH */
	  strcat (HDRname, HDR2ext);
	  Offptr = &vhOff;
	  printf ("\nDumping [VH] headers to [%s]", HDRname);
	  break;
	case 3:		/* VV */
	  strcat (HDRname, HDR3ext);
	  Offptr = &vvOff;
	  printf ("\nDumping [VV] headers to [%s]", HDRname);
	  break;
	}

      retv = dumpHdr (ADTSfp, HDRname, Offptr);
      if (retv != 0)
	{
	  fprintf (stderr,
		   "\n\nError[%s]: Error in dumping headers!", argv[0]);
	  fprintf (stderr, "\nExiting [convert884] program...\n\n");
	  fclose (ADTSfp);	/* Close input file....                    */
	  fclose (OUTfp);	/* Close output file..                     */
	  exit (1);
	}
    }
  else
    {
      printf ("\nNo headers dumped...");
    }

  /* Cleanup -- Close files */


  fclose (ADTSfp);		/* Close input file....                    */
  fclose (OUTfp);		/* Close output file..                     */

  switch (datatype)
    {
    case 0:			/* RAW COMPLEX */
      printf ("\nConversion to RAW complex image completed...\n\n");
      break;
    case 1:			/* RAW MAGNITUDE */
      printf ("\nConversion to RAW magnitude image completed...\n\n");
      break;
    case 2:			/* RAW MAGNITUDE-SQUARED */
      printf ("\nConversion to RAW magnitude**2 [sigma-0] image completed...\n\n");
      break;
    case 3:			/* RAW RCS */
      printf ("\nConversion to RAW RCS image completed...\n\n");
      break;
    }

}

/******************** LAST LINE OF convert884() *****************************/
