/*=========================================================================
 *=========================================================================
 == [DISCLAIMER]: THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS   ==
 == RELEASED "AS IS".  THE U.S. GOVERNMENT MAKES NO WARRANTY OF ANY     ==
 == KIND, EXPRESS OR IMPLIED, CONCERNING THIS SOFTWARE AND ANY          ==
 == ACCOMPANYING DOCUMENTATION, INCLUDING, WITHOUT LIMITATION, ANY      ==
 == WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  ==
 == IN NO EVENT WILL THE U.S. GOVERNMENT BE LIABLE FOR ANY DAMAGES      ==
 == ARISING OUT OF THE USE, OR INABILITY TO USE, THIS SOFTWARE OR ANY   ==
 == ACCOMPANYING DOCUMENTATION, EVEN IF INFORMED IN ADVANCE OF THE      ==
 == POSSIBLITY OF SUCH DAMAGES.                                         ==
 =========================================================================
 =========================================================================*/

/*************************************************************************
 *                                                                       *
 *                      Routine:  cv884.c                                *
 *                       Author:  J. Querns, Veda Inc.                   *
 *                         Date:  26 Feb 96                              *
 *                                                                       *
 *  Description: This routine converts a single-polarization 8-8-4       *
 *               formatted image to a RAW, headerless COMPLEX image,     *
 *               with each COMPLEX pixel = 4 bytes real + 4 bytes imag.  *
 *                                                                       *
 *               The command syntax is:                                  *
 *                                                                       *
 *                     cv884 <infile> <outfile>                          *
 *                                                                       *
 * Note:  Because of the way "ex884" extracts and saves single image     *
 *        8-8-4 files, the start of the image will always be the sum of  *
 *        the record length and the header-2 length (i.e. rl+hdr2len)..  *
 *                                                                       *
 *************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Define COMPLEX data structure */

struct COMPLEXN
  {
    float re;
    float im;
  };

/* Define 8-8-4 Pixel structure */

struct PIX884
  {
    char e;
    char i;
    char q;
  };



main (argc, argv)

     int argc;
     char **argv;

{
  FILE *outfp;			/* Ptr to output RAW image...     */
  FILE *infp;			/* Ptr to 8-8-4 file....          */
  char *INname;			/* 8-8-4 filename...              */
  char *OUTname;		/* RAW output filename..          */

  struct COMPLEXN *crec;	/* Ptr to array of complex num..  */
  struct PIX884 *rec884;	/* Ptr to array of 884 pixels...  */

  long l, s;

  char tbuff[9];
  long HDR2off;			/* Offset location of hdr2len..   */
  long IMAGEoff;		/* Image offset location...       */
  long nsamps;			/* Num of samples per image line. */
  long nlines;			/* Num of lines per image..       */
  long hdr2len;			/* Header-2 length...             */
  long rl;			/* Record length...               */

  /* Check for correct arguments */

  if (argc < 3)
    {
      fprintf (stderr, "\n\n[Convert 8-8-4 image to Complex RAW]");
      fprintf (stderr,
	       "\nUsage: cv884 <884 infile> <Complex RAW outfile>\n\n");
      exit (1);
    }

  INname = argv[1];		/* Retrieve 8-8-4 image input name           */
  OUTname = argv[2];		/* Retrieve RAW complex image output name    */


  /* Open 8-8-4 file for Reading */

  infp = fopen (INname, "rb");
  if (infp == NULL)
    {
      fprintf (stderr, "\n\nError[cv884]: Unable to open file: [%s]", INname);
      fprintf (stderr, "\nExiting [cv884] program...\n\n");
      exit (1);			/* FAILURE */
    }

  printf ("\n\nConvert 8-8-4 image to RAW complex....\n");


/****************************************************************************
 *                   RETRIEVE, PROCESS, & WRITE OUT DATA                    *
 ****************************************************************************/

/* Get HEADER-2 LENGTH  */
  HDR2off = 32L;
  if (fseek (infp, HDR2off, 0) != 0)
    {
      fprintf (stderr,
	  "\n\n[cv884]: Unable to seek to header-2 offset for image..\n\n");
      fclose (infp);
      exit (1);
    }
  else
    {
      fgets (tbuff, 9, infp);
      hdr2len = atol (tbuff);
    }

/* Get RECORD LENGTH  */
  if (fseek (infp, 40L, 0) != 0)
    {
      fprintf (stderr,
	    "\n\n[cv884]: Unable to seek to reclen offset for image..\n\n");
      fclose (infp);
      exit (1);
    }
  else
    {
      fgets (tbuff, 9, infp);
      rl = atol (tbuff);
    }

/* Get NSAMPS PER LINE */
  if (fseek (infp, 80L, 0) != 0)
    {
      fprintf (stderr,
	   "\n\n[cv884]: Unable to seek to samples offset for image..\n\n");
      fclose (infp);
      exit (1);
    }
  else
    {
      fgets (tbuff, 9, infp);
      nsamps = atol (tbuff);
    }

/* Get NLINES PER IMAGE */
  if (fseek (infp, 88L, 0) != 0)
    {
      fprintf (stderr,
	     "\n\n[cv884]: Unable to seek to lines offset for image..\n\n");
      fclose (infp);
      exit (1);
    }
  else
    {
      fgets (tbuff, 9, infp);
      nlines = atol (tbuff);
    }

/* Calculate location of sub-frame image 0 */
  IMAGEoff = rl + hdr2len;


  rewind (infp);		/* Rewind ADTS image file...          */

/*********************************
 * Allocate array memories...    *
 *********************************/

  rec884 = (struct PIX884 *) malloc (nsamps * sizeof (struct PIX884));

  if (rec884 == NULL)
    {
      fprintf (stderr,
	       "\n\nError[%s]: Unable to allocate 8-8-4 record memory!",
	       argv[0]);
      fprintf (stderr, "\nExiting [cv884] program...\n\n");
      fclose (infp);		/* Close input file */
      exit (1);
    }

  crec = (struct COMPLEXN *) malloc (nsamps * sizeof (struct COMPLEXN));

  if (crec == NULL)
    {
      fprintf (stderr,
	       "\n\nError[%s]: Unable to allocate complex number memory!",
	       argv[0]);
      fprintf (stderr, "\nExiting [cv884] program...\n\n");
      free (rec884);		/* Free memory      */
      fclose (infp);		/* Close input file */
      exit (1);
    }

/*********************************
 * Open output file for writing  *
 *********************************/

  outfp = fopen (OUTname, "wb");
  if (outfp == NULL)
    {
      fprintf (stderr, "\n\nError: Unable to create file: [%s]", OUTname);
      fprintf (stderr, "\nExiting [cv884] program...\n\n");
      free (rec884);		/* Free memory       */
      free (crec);		/* Free memory       */
      fclose (infp);		/* Close input file  */
      exit (1);
    }

  fseek (infp, IMAGEoff, 0);	/* Seek to start of image...      */
  for (l = 0; l < nlines; l++)
    {
      fseek (infp, 9L, 1);	/* Skip over three header pixels..   */
      fread (rec884, sizeof (struct PIX884), nsamps, infp);

      for (s = 0; s < nsamps; s++)
	{
	  crec[s].re = (float) ((long) rec884[s].i << (rec884[s].e & 017)) / 4096.0;
	  crec[s].im = (float) ((long) rec884[s].q << (rec884[s].e & 017)) / 4096.0;
	}
      fwrite (crec, sizeof (struct COMPLEXN), nsamps, outfp);

      fseek (infp, 3L, 1);	/* Skip over 1 trailer pixel...       */

    }				/* End of nlines loop */

  /* Cleanup -- Free memory & Close files */

  free (rec884);
  free (crec);

  fclose (infp);		/* Close input file....                    */
  fclose (outfp);		/* Close output raw binary file..          */

  printf ("\nConversion completed...\n\n");

}

/******************** LAST LINE OF cv884() *****************************/
