/*=========================================================================
 *=========================================================================
 == [DISCLAIMER]: THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS   ==
 == RELEASED "AS IS".  THE U.S. GOVERNMENT MAKES NO WARRANTY OF ANY     ==
 == KIND, EXPRESS OR IMPLIED, CONCERNING THIS SOFTWARE AND ANY          ==
 == ACCOMPANYING DOCUMENTATION, INCLUDING, WITHOUT LIMITATION, ANY      ==
 == WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  ==
 == IN NO EVENT WILL THE U.S. GOVERNMENT BE LIABLE FOR ANY DAMAGES      ==
 == ARISING OUT OF THE USE, OR INABILITY TO USE, THIS SOFTWARE OR ANY   ==
 == ACCOMPANYING DOCUMENTATION, EVEN IF INFORMED IN ADVANCE OF THE      ==
 == POSSIBLITY OF SUCH DAMAGES.                                         ==
 =========================================================================
 =========================================================================*/

/*************************************************************************
 *                                                                       *
 *                      Routine:  ex884.c                                *
 *                       Author:  J. Querns, Veda Inc.                   *
 *                                5200 Springfield Pike, Suite 200       *
 *                                Dayton, OH 45431                       *
 *                                (937) 255-1116, Ext 2818               *
 *                                                                       *
 *                         Date:  3 Jun 97                               *
 *                                                                       *
 *  Description:  This routine reads an MIT/LL ADTS (ATRWG) 8-8-4 for-   *
 *                matted file, extracts a user-specified polarization    *
 *                sub-image (HH, HV, VH, or VV), and writes it back out  *
 *                in 8-8-4 format as a single 8-8-4 file+header.         *
 *                                                                       *
 *                This routine ALWAYS uses the header-1 and header-2     *
 *                from the HH sub-image, except that it changes the      *
 *                "dataname" in the headers to reflect the specific      *
 *                sub-image polarization selected.  For example, if the  *
 *                VH polarization is selected and the dataname in the HH *
 *                header-1 field = "P8F4HH", then the dataname will be   *
 *                updated to "P8F4VH"...                                 *
 *                                                                       *
 *                NOTE: There is NO DEFAULT!! You must specify the pol-  *
 *                      arization to extract to 8-8-4 format.  Also, this*
 *                      routine EXPECTS input file to have 4 sub-images. *
 *                                                                       *
 *                The command syntax is:                                 *
 *                                                                       *
 *                     ex884 -i <infile>        {REQUIRED}               *
 *                           -o <outfile>       {REQUIRED}               *
 *                                                                       *
 *                     -p<popt>  {Polarization option}                   *
 *                                                                       *
 *                         0: HH polarization                            *
 *                         1: HV polarization                            *
 *                         2: VH polarization                            *
 *                         3: VV polarization                            *
 *                                                                       *
 *                                                                       *
 *  <ADTS FILE FORMAT>:                                                  *
 *                                                                       *
 *  For information on the ADTS 8-8-4 File format, read the file,        *
 *  "ADTS_README"...                                                     *
 *                                                                       *
 * Calls: getImgLoc() -- This subroutine returns the file location of    *
 *                       the specified polarization sub-image, the sub-  *
 *                       image header-1, the mission header (header-2),  *
 *                       and the number of samples and lines...          *
 *                                                                       *
 *      read_switch() -- Extract command line arguments...               *
 *                                                                       *
 *************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "adts.h"

extern int getImgLoc ();	/* Extracts img & hdr locations   */
extern char *read_switch ();	/* Extracts arguments...          */

main (argc, argv)

     int argc;
     char *argv[];

{
  FILE *OUTfp;			/* Ptr to output OUT image...     */
  FILE *ADTSfp;			/* Ptr to ADTS file....           */
  char *ADTSname;		/* ADTS filename...               */
  char *OUTname;		/* OUT filename...                */

  struct ADTSPIX *adtsrec;	/* Ptr to array of ADTS pixels... */
  struct ADTSImgOff *Offptr;	/* Ptr to sub-image offset struct */

  char *hdr1, *hdr2;
  int dlen;
  long DNoff;			/* Pos of dataname...             */

  long i, l, s, bpl, n;
  long linesize;		/* Size of one 8-8-8 line (ns+4)  */
  long ns, nl;			/* Num of lines and samples in img */

  int retv;			/* Return value (i.e. status)..   */
  int poltype;			/* Polarization image type wanted */
  char *dpos;
  char *dptr;
  char dataname[9];		/* Holds dataname of sub-image..  */

  char *POLopt;			/* Ptr to variable pol option     */
  char *POLopt0;		/* Ptr to HH pol selection        */
  char *POLopt1;		/* Ptr to HV pol selection        */
  char *POLopt2;		/* Ptr to VH pol selection        */
  char *POLopt3;		/* Ptr to VV pol selection        */

  /* Check for correct arguments */

  if (argc < 2)
    {
      fprintf (stderr, "\n\n[EXTRACT 8-8-4 Sub-Image]:");
      fprintf (stderr, "\nUsage: ex884 [<switches>]\n");
      fprintf (stderr, "\n   -i <ifile> --> [Input ADTS 4-polarization file]");
      fprintf (stderr, "\n   -o <ofile> --> [Output 8-8-4 sub-image file]");
      fprintf (stderr,
	  "\n   -p <popt > --> [Polarization: 0=HH, 1=HV, 2=VH, 3=VV]\n\n");
      exit (1);
    }
  /* Extract Optional Switch Arguments */

  ADTSname = read_switch (&argc, argv, "-i", 1, NULL);
  OUTname = read_switch (&argc, argv, "-o", 1, NULL);

/****************************************** 
 * Check for ADTS (8-8-4) Input file name *
 ******************************************/

  if (ADTSname == (char *) NULL)
    {
      fprintf (stderr, "\n\nError: no input file entered or incorrect syntax!");
      fprintf (stderr, "\nInput file syntax = -i <infile>\n\n");
      exit (1);
    }
  else
    {
      printf ("\n\nInput  file: [%s]", ADTSname);
    }

/******************************
 * Check for Output file name *
 ******************************/

  if (OUTname == (char *) NULL)
    {
      fprintf (stderr, "\n\nError: no output file entered or incorrect syntax!");
      fprintf (stderr, "\nOutput file syntax = -o <outfile>\n\n");
      exit (1);
    }
  else
    {
      printf ("\nOutput file: [%s]", OUTname);
    }

/****
 This extracts POL options in one of two formats:
 "-p <popt>" or "-p<popt>". Note that the 1st format
 is has a space between the switch and the argument.
 The 2nd format does not.  The 2nd format was added
 to allow the user to select the format desired without
 worrying about fumbling with the keys and forgetting
 to enter a space between the switch and the argument,
 as you must do with the 1st format...
 ****/

  POLopt = read_switch (&argc, argv, "-p", 1, NULL);
  POLopt0 = read_switch (&argc, argv, "-p0", 0, NULL);
  POLopt1 = read_switch (&argc, argv, "-p1", 0, NULL);
  POLopt2 = read_switch (&argc, argv, "-p2", 0, NULL);
  POLopt3 = read_switch (&argc, argv, "-p3", 0, NULL);

/**************************************
 * Check for polarization type option *
 **************************************/

/* Check for POLopt format-2 ... */
  if (POLopt0 != (char *) NULL)
    {
      poltype = 0;		/* HH selected */
    }
  else if (POLopt1 != (char *) NULL)
    {
      poltype = 1;		/* HV selected */
    }
  else if (POLopt2 != (char *) NULL)
    {
      poltype = 2;		/* VH selected */
    }
  else if (POLopt3 != (char *) NULL)
    {
      poltype = 3;		/* VV selected */
    }
  else
    {
      poltype = atoi (POLopt);
      if ((poltype < 0) || (poltype > 3))
	{
	  fprintf (stderr, "\n\nError: invalid polarization type!");
	  fprintf (stderr, "\n  -p <popt> ");
	  fprintf (stderr, "\n     0: HH Sub-Image");
	  fprintf (stderr, "\n     1: HV Sub-Image");
	  fprintf (stderr, "\n     2: VH Sub-Image");
	  fprintf (stderr, "\n     3: VV Sub-Image\n\n");
	  exit (1);
	}
    }

  switch (poltype)
    {
    case 0:			/* HH */
      printf ("\nExtracting [HH] polarization...");
      break;
    case 1:			/* HV */
      printf ("\nExtracting [HV] polarization...");
      break;
    case 2:			/* VH */
      printf ("\nExtracting [VH] polarization...");
      break;
    case 3:			/* VV */
      printf ("\nExtracting [VV] polarization...");
      break;
    }

  /* Open ADTS file for Reading */

  ADTSfp = fopen (ADTSname, "rb");
  if (ADTSfp == NULL)
    {
      fprintf (stderr, "\n\nError[ex884]: Unable to open file: [%s]", ADTSname);
      fprintf (stderr, "\nExiting [ex884] program...\n\n");
      exit (1);			/* FAILURE */
    }

/****************************************************************************
 *                     Get Sub-Image Offset Location Data                   *
 ****************************************************************************/

  if ((retv = getImgLoc (ADTSfp)) != 0)
    {
      fprintf (stderr, "\n\nError[ex884]: in obtaining offset location data");
      fprintf (stderr, "\nExiting [ex884] program...\n\n");
      fclose (ADTSfp);
      exit (1);			/* FAILURE */
    }

  switch (poltype)
    {
    case 0:			/* HH */
      Offptr = &hhOff;
      break;
    case 1:			/* HV */
      Offptr = &hvOff;
      break;
    case 2:			/* VH */
      Offptr = &vhOff;
      break;
    case 3:			/* VV */
      Offptr = &vvOff;
      break;
    }

  ns = Offptr->nsamps;
  nl = Offptr->nlines;

/****************************************************************************
 *                   RETRIEVE, PROCESS, & WRITE OUT DATA                    *
 ****************************************************************************/


/*************************************
 * Allocate ADTS record memory...    *
 *************************************/

  linesize = ns + 4;		/* Number of data samples + 4 hdr & trlr pixels */
  adtsrec = (struct ADTSPIX *) malloc (linesize * sizeof (struct ADTSPIX));

  if (adtsrec == NULL)
    {
      fprintf (stderr,
	       "\n\nError[ex884]: Unable to allocate ADTS record memory!");
      fprintf (stderr, "\nExiting [ex884] program...\n\n");
      fclose (ADTSfp);		/* Close input file */
      exit (1);
    }

/***************************************
 * Open output image file for writing  *
 ***************************************/

  OUTfp = fopen (OUTname, "wb");
  if (OUTfp == NULL)
    {
      fprintf (stderr, "\n\nError[ex884]: Unable to create file: [%s]", OUTname);
      fprintf (stderr, "\nExiting [ex884] program...\n\n");
      free (adtsrec);		/* Free memory       */
      fclose (ADTSfp);		/* Close input file  */
      exit (1);
    }

/**************************************************************************
 *                 Retrieve Sub-Image Header-1 & write-out                *
 * Use HH Header-1 & Header-2 for all other polarizations, except change  *
 * dataname to reflect specific polarization sub-image...                 *
 * Retrieve dataname for specific sub-image first...                      *
 **************************************************************************/

  DNoff = Offptr->hdr1 + 16L;
  fseek (ADTSfp, DNoff, 0);	/* Starting point of specific sub-image */
  fgets (dataname, 9, ADTSfp);	/* Retrieve sub-image dataname...       */

  fseek (ADTSfp, hhOff.hdr1, 0);	/* Go to starting pos. of HH header-1   */
  hdr1 = (char *) malloc (hhOff.hdr1len * sizeof (char));

  if (hdr1 == NULL)
    {
      fprintf (stderr,
	       "\n\nError[ex884]: Unable to allocate header-1 memory!");
      free (adtsrec);		/* Free memory       */
      fclose (ADTSfp);		/* Close input file  */
      exit (1);
    }
  /* Retrieve  HH header-1...             */
  n = fread (hdr1, sizeof (char), hhOff.hdr1len, ADTSfp);

  if (n != hhOff.hdr1len)
    {
      fprintf (stderr,
	       "\n\nError[ex884]: in retrieving Header-1 data!");
      fprintf (stderr,
	       "\nOnly retrieved [%ld] of [%ld] bytes..\n\n",
	       n, hhOff.hdr1len);
      free (hdr1);
      free (adtsrec);		/* Free memory       */
      fclose (ADTSfp);		/* Close input file  */
    }

  dpos = hdr1 + 16;		/* Pt to pos of dataname in HH header-1 array */
  dptr = dataname;		/* Copy dataname for sub-image to header-1 array */
  dlen = strlen (dptr);
  strncpy (dpos, dptr, dlen);

  printf ("\nWriting Header-1...");
  n = fwrite (hdr1, sizeof (char), hhOff.hdr1len, OUTfp);

  if (n != hhOff.hdr1len)
    {
      fprintf (stderr,
	       "\n\nError[ex884]: Error in writing header-1!");
      fprintf (stderr, "\nOnly wrote [%d] of [%d] bytes\n\n", n, hhOff.hdr1len);
      free (hdr1);
      fclose (ADTSfp);		/* Close input file....                    */
      fclose (OUTfp);		/* Close output file..                     */
      free (adtsrec);
      exit (1);
    }
  free (hdr1);

/**************************************************************************
 *           Retrieve Sub-Image [Mission] Header-2 & write-out            *
 **************************************************************************/

  fseek (ADTSfp, hhOff.hdr2, 0);
  hdr2 = (char *) malloc (hhOff.hdr2len * sizeof (char));

  if (hdr2 == NULL)
    {
      fprintf (stderr,
	       "\n\nError[ex884]: Unable to allocate header-2 memory!");
      free (adtsrec);		/* Free memory       */
      fclose (ADTSfp);		/* Close input file  */
      exit (1);
    }

  n = fread (hdr2, sizeof (char), hhOff.hdr2len, ADTSfp);

  if (n != hhOff.hdr2len)
    {
      fprintf (stderr,
	       "\n\nError[ex884]: in retrieving Header-2 data!");
      fprintf (stderr,
	       "\nOnly retrieved [%ld] of [%ld] bytes..\n\n",
	       n, hhOff.hdr2len);
      free (hdr2);
      free (adtsrec);		/* Free memory       */
      fclose (ADTSfp);		/* Close input file  */
    }

  printf ("\nWriting [Mission] Header-2...");
  n = fwrite (hdr2, sizeof (char), hhOff.hdr2len, OUTfp);

  if (n != hhOff.hdr2len)
    {
      fprintf (stderr,
	       "\n\nError[ex884]: Error in writing header-2!");
      fprintf (stderr, "\nOnly wrote [%d] of [%d] bytes\n\n", n, hhOff.hdr2len);
      free (hdr2);
      fclose (ADTSfp);		/* Close input file....                    */
      fclose (OUTfp);		/* Close output file..                     */
      free (adtsrec);
      exit (1);
    }
  free (hdr2);

/*********************************
 *   Process Data....            *
 *********************************
 * Each fseek in 1st switch seeks*
 * to start of next ADTS record. *
 *********************************/

  switch (poltype)
    {
    case 0:			/* HH */
      printf ("\nWriting 8-8-4 [HH] Sub-image...");
      break;
    case 1:			/* HV */
      printf ("\nWriting 8-8-4 [HV] Sub-image...");
      break;
    case 2:			/* VH */
      printf ("\nWriting 8-8-4 [VH] Sub-image...");
      break;
    case 3:			/* VV */
      printf ("\nWriting 8-8-4 [VV] Sub-image...");
      break;
    }

  rewind (ADTSfp);		/* Rewind ADTS image file...           */
  bpl = (ns + 4) * 3;		/* Bytes per line (inc. hdr&tr pixels) */
  for (l = 0; l < nl; l++)
    {
      switch (poltype)
	{
	case 0:		/* HH */
	  fseek (ADTSfp, (hhOff.img) + (l * bpl), 0);
	  break;
	case 1:		/* HV */
	  fseek (ADTSfp, (hvOff.img) + (l * bpl), 0);
	  break;
	case 2:		/* VH */
	  fseek (ADTSfp, (vhOff.img) + (l * bpl), 0);
	  break;
	case 3:		/* VV */
	  fseek (ADTSfp, (vvOff.img) + (l * bpl), 0);
	  break;
	}

      n = fread (adtsrec, 1, bpl, ADTSfp);
      if (n != bpl)
	{
	  fprintf (stderr,
		   "\n\nError[ex884]: Error in reading line [%d]!", l);
	  fprintf (stderr, "\nOnly read [%d] of [%d] bytes\n\n", n, bpl);
	  fclose (ADTSfp);	/* Close input file....                    */
	  fclose (OUTfp);	/* Close output file..                     */
	  free (adtsrec);
	  exit (1);
	}

      n = fwrite (adtsrec, 1, bpl, OUTfp);
      if (n != bpl)
	{
	  fprintf (stderr,
		   "\n\nError[ex884]: Error in writing line [%d]!", l);
	  fprintf (stderr, "\nOnly wrote [%d] of [%d] bytes\n\n", n, bpl);
	  fclose (ADTSfp);	/* Close input file....                    */
	  fclose (OUTfp);	/* Close output file..                     */
	  free (adtsrec);
	  exit (1);
	}

    }				/* End of processing loop */


  /* Cleanup -- Close files */

  free (adtsrec);		/* Free ADTS image record memory...        */
  fclose (ADTSfp);		/* Close input file....                    */
  fclose (OUTfp);		/* Close output file..                     */

  switch (poltype)
    {
    case 0:			/* HH */
      printf ("\nExtraction of [HH] 8-8-4 sub-image completed...\n\n");
      break;
    case 1:			/* HV */
      printf ("\nExtraction of [HV] 8-8-4 sub-image completed...\n\n");
      break;
    case 2:			/* VH */
      printf ("\nExtraction of [VH] 8-8-4 sub-image completed...\n\n");
      break;
    case 3:			/* VV */
      printf ("\nExtraction of [VV] 8-8-4 sub-image completed...\n\n");
      break;
    }

}

/******************** LAST LINE OF ex884() *****************************/
