/*=========================================================================
 *=========================================================================
 == [DISCLAIMER]: THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS   ==
 == RELEASED "AS IS".  THE U.S. GOVERNMENT MAKES NO WARRANTY OF ANY     ==
 == KIND, EXPRESS OR IMPLIED, CONCERNING THIS SOFTWARE AND ANY          ==
 == ACCOMPANYING DOCUMENTATION, INCLUDING, WITHOUT LIMITATION, ANY      ==
 == WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  ==
 == IN NO EVENT WILL THE U.S. GOVERNMENT BE LIABLE FOR ANY DAMAGES      ==
 == ARISING OUT OF THE USE, OR INABILITY TO USE, THIS SOFTWARE OR ANY   ==
 == ACCOMPANYING DOCUMENTATION, EVEN IF INFORMED IN ADVANCE OF THE      ==
 == POSSIBLITY OF SUCH DAMAGES.                                         ==
 =========================================================================
 =========================================================================*/

/*************************************************************************
 *                                                                       *
 *                      Routine:  raw884cnv.c                            *
 *                       Author:  J. Querns, Veda Inc.                   *
 *                         Date:  3 Jun 97                               *
 *                                                                       *
 *  Description: This routine converts a RAW, headerless image in 8-8-4  *
 *               format to RAW complex format..                          *
 *                                                                       * 
 *               Every 8-8-4 image line is preceded by 3 header pixels   *
 *               and followed by 1 trailer pixel.  Each 8-8-4 pixel      *
 *               (including the header & trailer pixels) is 3 bytes      *
 *               long.  The header pixels contain control information    *
 *               (frame, line number, polarization, etc.).  The image    *
 *               pixels have the following breakdown:                    *
 *                                                                       *
 *               Byte 1: 4-bit unsigned exponent (ignore 1st 4 bits)     *
 *               Byte 2: Inphase (or real) integer mantissa              *
 *               Byte 3: Quadrature (or imaginary) integer mantissa      *
 *                                                                       *
 *               The VALUE of each mantissa is computed by:              *
 *                                                                       *
 *               VALUE = SCALE X MANTISSA X 2**exponent                  *
 *                                                                       *
 *                       where -> SCALE = 4096 e-01                      *
 *                                                                       * 
 *   Procedure:  The main processing loop goes like this:                *
 *                                                                       *
 *               For each line                                           *
 *                  {                                                    *
 *                   Skip over first 3 pixels (un-needed hdr pixels)     *
 *                   Read "nsamps" 8-8-4 pixel data into array           *
 *                   Convert "nsamps" pixel data to COMPLEX data         *
 *                   Write "nsamps" COMPLEX pixel data to output file    *
 *                   Skip over trailer pixel (at end of normal line)     *
 *                  }                                                    *
 *                                                                       *
 *                                                                       *
 *************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/* Define COMPLEX data structure */

struct COMPLEXN
  {
    float re;
    float im;
  };

/* Define ADTS 8-8-4 Pixel structure */

struct ADTSPIX
  {
    char e;
    char i;
    char q;
  };



main (argc, argv)

     int argc;
     char **argv;

{
  FILE *outfp;			/* Ptr to output RAW image...     */
  FILE *infp;			/* Ptr to ADTS file....           */
  char *INname;			/* ADTS filename...               */
  char *OUTname;		/* RAW filename...                */

  struct COMPLEXN *crec;	/* Ptr to array of complex num..  */
  struct ADTSPIX *adtsrec;	/* Ptr to array of ADTS pixels... */

  long l, s;

  long nsamps;			/* Num of samples per image line. */
  long nlines;			/* Num of lines per image..       */

  /* Check for correct arguments */

  if (argc < 5)
    {
      fprintf (stderr,
       "\n\nUsage: %s <inname> <outname> <nsamples> <nlines>\n\n", argv[0]);
      exit (1);
    }

  INname = argv[1];		/* Retrieve 8-8-4 image input name           */
  OUTname = argv[2];		/* Retrieve RAW complex image output name    */
  nsamps = atol (argv[3]);	/* Retrieve number of samples per image line */
  nlines = atol (argv[4]);	/* Retrieve number of image lines            */

  /* Open 8-8-4 file for Reading */

  infp = fopen (INname, "rb");
  if (infp == NULL)
    {
      fprintf (stderr, "\n\nError: Unable to open file: [%s]", INname);
      fprintf (stderr, "\nExiting [raw884cnv] program...\n\n");
      exit (1);			/* FAILURE */
    }

  printf ("\n\nConvert RAW 8-8-4 image to RAW complex....\n");
  printf ("\nNumber of samples per image line = [%ld]", nsamps);
  printf ("\nNumber of lines per image        = [%ld]\n", nlines);


/****************************************************************************
 *                   RETRIEVE, PROCESS, & WRITE OUT DATA                    *
 ****************************************************************************/

  rewind (infp);		/* Rewind ADTS image file...          */

/*********************************
 * Allocate array memories...    *
 *********************************/

  adtsrec = (struct ADTSPIX *) malloc (nsamps * sizeof (struct ADTSPIX));

  if (adtsrec == NULL)
    {
      fprintf (stderr,
	       "\n\nError[%s]: Unable to allocate ADTS record memory!",
	       argv[0]);
      fprintf (stderr, "\nExiting [raw884cnv] program...\n\n");
      fclose (infp);		/* Close input file */
      exit (1);
    }

  crec = (struct COMPLEXN *) malloc (nsamps * sizeof (struct COMPLEXN));

  if (crec == NULL)
    {
      fprintf (stderr,
	       "\n\nError[%s]: Unable to allocate complex number memory!",
	       argv[0]);
      fprintf (stderr, "\nExiting [adt2raw] program...\n\n");
      free (adtsrec);		/* Free memory      */
      fclose (infp);		/* Close input file */
      exit (1);
    }

/*********************************
 * Open output file for writing  *
 *********************************/

  outfp = fopen (OUTname, "wb");
  if (outfp == NULL)
    {
      fprintf (stderr, "\n\nError: Unable to create file: [%s]", OUTname);
      fprintf (stderr, "\nExiting [raw884cnv] program...\n\n");
      free (adtsrec);		/* Free memory       */
      free (crec);		/* Free memory       */
      fclose (infp);		/* Close input file  */
      exit (1);
    }

  for (l = 0; l < nlines; l++)
    {
      fseek (infp, 9L, 1);	/* Skip over three header pixels..   */
      fread (adtsrec, sizeof (struct ADTSPIX), nsamps, infp);

      for (s = 0; s < nsamps; s++)
	{
	  crec[s].re = (float) ((long) adtsrec[s].i << (adtsrec[s].e & 017)) / 4096.0;
	  crec[s].im = (float) ((long) adtsrec[s].q << (adtsrec[s].e & 017)) / 4096.0;
	}
      fwrite (crec, sizeof (struct COMPLEXN), nsamps, outfp);

      fseek (infp, 3L, 1);	/* Skip over 1 trailer pixel...       */

    }				/* End of nlines loop */

  /* Cleanup -- Free memory & Close files */

  free (adtsrec);
  free (crec);

  fclose (infp);		/* Close input file....                    */
  fclose (outfp);		/* Close output raw binary file..          */

  printf ("\nConversion completed...\n\n");

}

/******************** LAST LINE OF raw884cnv() *****************************/
