/*=========================================================================
 *=========================================================================
 == [DISCLAIMER]: THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS   ==
 == RELEASED "AS IS".  THE U.S. GOVERNMENT MAKES NO WARRANTY OF ANY     ==
 == KIND, EXPRESS OR IMPLIED, CONCERNING THIS SOFTWARE AND ANY          ==
 == ACCOMPANYING DOCUMENTATION, INCLUDING, WITHOUT LIMITATION, ANY      ==
 == WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  ==
 == IN NO EVENT WILL THE U.S. GOVERNMENT BE LIABLE FOR ANY DAMAGES      ==
 == ARISING OUT OF THE USE, OR INABILITY TO USE, THIS SOFTWARE OR ANY   ==
 == ACCOMPANYING DOCUMENTATION, EVEN IF INFORMED IN ADVANCE OF THE      ==
 == POSSIBLITY OF SUCH DAMAGES.                                         ==
 =========================================================================
 =========================================================================*/

/*************************************************************************
 *                                                                       *
 *            Subroutine:  getImgLoc()                                   *
 *                Author:  J. Querns, Veridian Inc (Veda)                *
 *                         5200 Springfield Pike, Suite 200              *
 *                         Dayton, OH 45431                              *
 *                         (937) 255-1116, Ext 2818                      *
 *                                                                       *
 *                  Date:  3 Jun 97                                      *
 *                                                                       *
 * DESCRIPTION:  This routine searches the file represented by the       *
 *               input ADTS file pointer for image and header offset     *
 *               location information and returns the data in appropriate*
 *               structure for each polarization sub-image.  Since it    *
 *               doesn't take very long, this return will extract        *
 *               location information for every sub-image in the ADTS    *
 *               file.                                                   *
 *                                                                       *
 *               This subroutine assumes that if the ADTS file is a      *
 *               standard file with 4 polarization sub-images, each      *
 *               image will have the same number of samples per image    *
 *               line and the same number of image lines.  It DOES NOT   *
 *               assume that each sub-image has the same number of       *
 *               records per sub-image.                                  *
 *                                                                       *
 * FORMAT:       int getImgLoc (infp)                                    *
 *                                                                       *
 *               [infp] -- FILE pointer to open ADTS input file.         *
 *                                                                       *
 *               The location information for each polarization is       *
 *               defined in a structure declared by the include file,    *
 *               "adts.h":                                               *
 *                                                                       *
 *                     struct ADTSImgOFF {                               *
 *                        long img;                                      *
 *                        long hdr1;                                     *
 *                        long hdr2;                                     *
 *                        long hdr2len;                                  *
 *                        long nsamps;                                   *
 *                        long nlines;                                   *
 *                        };                                             *
 *                                                                       *
 *                     struct ADTSImgOff hhOff;  (HH Pol)                *
 *                     struct ADTSImgOff hvOff;  (HV Pol)                *
 *                     struct ADTSImgOff vhOff;  (VH Pol)                *
 *                     struct ADTSImgOff vvOff;  (VV Pol)                *
 *                                                                       *
 *   NOTE:  "hdr2len" must be retrieved, if available...                 *
 *                                                                       *
 * RETURN:       0, if successful                                        *
 *               1, if failure...                                        *
 *                                                                       *
 *************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include "adts.h"


int 
getImgLoc (infp)

     FILE *infp;

{
  char tbuff[9];

  long nlines, nsamps;
  long nr0;			/* nrecs for HH pol type..        */
  long rl0;			/* Bytes per record for HH pol    */

  long hdr2len0;		/* hdr2len for HH pol             */
  long hdr2len1;		/* hdr2len for HV pol             */
  long hdr2len2;		/* hdr2len for VH pol             */
  long hdr2len3;		/* hdr2len for VV pol             */

  long nr1, nr2, nr3;		/* nrecs for other pols, if there */
  long rl1, rl2, rl3;		/* Bytes per rec for other pols.. */

  long NRoff;			/* Offset location for nrecs...   */
  long RLoff;			/* Offset location for reclen...  */
  long HDR2off;			/* Offset location for hdr2len... */

  long IMAGEoff0;
  long IMAGEoff1;
  long IMAGEoff2;
  long IMAGEoff3;

  long nbytes0;			/* Numbytes for whole sub-image 0 */
  long nbytes1;			/* Numbytes for whole sub-image 1 */
  long nbytes2;			/* Numbytes for whole sub-image 2 */
  long nbytes3;			/* Numbytes for whole sub-image 3 */


	     /******************************************
	      * Find num of samples per image line and *
	      * number of lines per image..these are   *
	      * common to each sub-frame..use 0th sub- *
	      * frame...                               *
	      ******************************************/

/* Find number of samples per line from image sub-frame 0 */
  if (fseek (infp, 80L, 0) != 0)
    {
      fprintf (stderr,
	       "\n\n[getImgLoc]: Unable to seek to samples offset for sub-frame 0..\n\n");
      return 1;
    }
  else
    {
      fgets (tbuff, 9, infp);
      nsamps = atol (tbuff);
    }

/* Find number of lines per image from image sub-frame 0 */
  if (fseek (infp, 88L, 0) != 0)
    {
      fprintf (stderr,
	       "\n\n[getImgLoc]: Unable to seek to lines offset for sub-frame 0..\n\n");
      return 1;
    }
  else
    {
      fgets (tbuff, 9, infp);
      nlines = atol (tbuff);
    }

	/**************************************************** 
	 *            Find Sub-Image Start Points           *
	 ****************************************************
	 * NOTE: Finds num of records and bytes per record  *
	 *       for each available polarization before     *
	 *       continuing....                             *
	 ****************************************************/


/***************************
 * SUB-FRAME IMAGE 0 - HH  *
 ***************************/


  NRoff = 24L;
  HDR2off = 32L;
  RLoff = 40L;

/* Find number of records from image sub-frame 0 */
  if (fseek (infp, NRoff, 0) != 0)
    {
      fprintf (stderr,
	       "\n\n[getImgLoc]: Unable to seek to nrecs offset for sub-frame 0..\n\n");
      return 1;
    }
  else
    {
      fgets (tbuff, 9, infp);
      nr0 = atol (tbuff);
    }

/* Find header-2 length from sub-frame 0 */
  if (fseek (infp, HDR2off, 0) != 0)
    {
      fprintf (stderr,
	       "\n\n[getImgLoc]: Unable to seek to hdr2len offset for sub-frame 0..\n\n");
      return 1;
    }
  else
    {
      fgets (tbuff, 9, infp);
      hdr2len0 = atol (tbuff);
    }

/* Find record length from image sub-frame 0 */
  if (fseek (infp, RLoff, 0) != 0)
    {
      fprintf (stderr,
	       "\n\n[getImgLoc]: Unable to seek to reclen offset for sub-frame 0..\n\n");
      return 1;
    }
  else
    {
      fgets (tbuff, 9, infp);
      rl0 = atol (tbuff);
    }

/* Calculate location of sub-frame image 0 */
  IMAGEoff0 = (nr0 - nlines) * rl0;

/* Calculate number of bytes for whole sub-frame image 0 */
  nbytes0 = nr0 * rl0;


/***************************
 * SUB-FRAME IMAGE 1 - HV  *
 ***************************/

  NRoff = (long) (nbytes0 + 24L);
  HDR2off = (long) (nbytes0 + 32L);
  RLoff = (long) (nbytes0 + 40L);

/* Find number of records from image sub-frame 1 */
  if (fseek (infp, NRoff, 0) != 0)
    {
      fprintf (stderr,
	       "\n\n[getImgLoc]: Unable to seek to nrecs offset for sub-frame 1..\n\n");
      return 1;
    }
  else
    {
      fgets (tbuff, 9, infp);
      nr1 = atol (tbuff);
    }

/* Find header-2 length from sub-frame 1 */
  if (fseek (infp, HDR2off, 0) != 0)
    {
      fprintf (stderr,
	       "\n\n[getImgLoc]: Unable to seek to hdr2len offset for sub-frame 1..\n\n");
      return 1;
    }
  else
    {
      fgets (tbuff, 9, infp);
      hdr2len1 = atol (tbuff);
    }

/* Find record length from image sub-frame 1 */
  if (fseek (infp, RLoff, 0) != 0)
    {
      fprintf (stderr,
	       "\n\n[getImgLoc]: Unable to seek to reclen offset for sub-frame 1..\n\n");
      return 1;
    }
  else
    {
      fgets (tbuff, 9, infp);
      rl1 = atol (tbuff);
    }

/* Calculate location of sub-frame image 1 */
  IMAGEoff1 = nbytes0 + ((nr1 - nlines) * rl1);

/* Calculate number of bytes for whole sub-image 1 */
  nbytes1 = nr1 * rl1;

/***************************
 * SUB-FRAME IMAGE 2 - VH  *
 ***************************/

  NRoff = (long) (nbytes0 + nbytes1 + 24L);
  HDR2off = (long) (nbytes0 + nbytes1 + 32L);
  RLoff = (long) (nbytes0 + nbytes1 + 40L);

/* Find number of records from image sub-frame 2 */
  if (fseek (infp, NRoff, 0) != 0)
    {
      fprintf (stderr,
	       "\n\n[getImgLoc]: Unable to seek to nrecs offset for sub-frame 2..\n\n");
      return 1;
    }
  else
    {
      fgets (tbuff, 9, infp);
      nr2 = atol (tbuff);
    }

/* Find header-2 length from sub-frame 2 */
  if (fseek (infp, HDR2off, 0) != 0)
    {
      fprintf (stderr,
	       "\n\n[getImgLoc]: Unable to seek to hdr2len offset for sub-frame 2..\n\n");
      return 1;
    }
  else
    {
      fgets (tbuff, 9, infp);
      hdr2len2 = atol (tbuff);
    }

/* Find record length from image sub-frame 2 */
  if (fseek (infp, RLoff, 0) != 0)
    {
      fprintf (stderr,
	       "\n\n[getImgLoc]: Unable to seek to reclen offset for sub-frame 2..\n\n");
      return 1;
    }
  else
    {
      fgets (tbuff, 9, infp);
      rl2 = atol (tbuff);
    }

/* Calculate location of sub-frame image 2 */
  IMAGEoff2 = nbytes0 + nbytes1 + ((nr2 - nlines) * rl2);

/* Calculate number of bytes for whole sub-image 2 */
  nbytes2 = nr2 * rl2;

/***************************
 * SUB-FRAME IMAGE 3 - VV  *
 ***************************/

  NRoff = (long) (nbytes0 + nbytes1 + nbytes2 + 24L);
  HDR2off = (long) (nbytes0 + nbytes1 + nbytes2 + 32L);
  RLoff = (long) (nbytes0 + nbytes1 + nbytes2 + 40L);

/* Find number of records from image sub-frame 3 */
  fseek (infp, NRoff, 0);
  if (fseek (infp, NRoff, 0) != 0)
    {
      fprintf (stderr,
	       "\n\n[getImgLoc]: Unable to seek to nrecs offset for sub-frame 3..\n\n");
      return 1;
    }
  else
    {
      fgets (tbuff, 9, infp);
      nr3 = atol (tbuff);
    }

/* Find header-2 length from sub-frame 3 */
  if (fseek (infp, HDR2off, 0) != 0)
    {
      fprintf (stderr,
	       "\n\n[getImgLoc]: Unable to seek to hdr2len offset for sub-frame 3..\n\n");
      return 1;
    }
  else
    {
      fgets (tbuff, 9, infp);
      hdr2len3 = atol (tbuff);
    }

/* Find record length from image sub-frame 3 */
  fseek (infp, RLoff, 0);
  if (fseek (infp, RLoff, 0) != 0)
    {
      fprintf (stderr,
	       "\n\n[getImgLoc]: Unable to seek to reclen offset for sub-frame 3..\n\n");
      return 1;
    }
  else
    {
      fgets (tbuff, 9, infp);
      rl3 = atol (tbuff);
    }

/* Calculate location of sub-frame image 3 */
  IMAGEoff3 = nbytes0 + nbytes1 + nbytes2 + ((nr3 - nlines) * rl3);

/* Calculate number of bytes for whole sub-image 3 */
  nbytes3 = nr3 * rl3;


   /*************************************************************
    * Store info in ADTS Offset Structure for each Polarization *
    *************************************************************/

/******************
 *  HH Sub-Image  *
 ******************/

  hhOff.img = IMAGEoff0;
  hhOff.hdr1 = 0L;

  if (hdr2len0 > 0)		/* A Header-2 exists for HH sub-image */
    {
      hhOff.hdr2 = rl0;
      hhOff.hdr2len = hdr2len0;
    }
  else
    /* No header-2... */
    {
      hhOff.hdr2 = 0L;
      hhOff.hdr2len = 0L;
    }
  hhOff.nsamps = nsamps;
  hhOff.nlines = nlines;


/******************
 *  HV Sub-Image  *
 ******************/

  hvOff.img = IMAGEoff1;
  hvOff.hdr1 = nbytes0;

  if (hdr2len1 > 0)		/* A Header-2 exists for HV sub-image */
    {
      hvOff.hdr2 = nbytes0 + rl1;
      hvOff.hdr2len = hdr2len1;
    }
  else
    /* No header-2... */
    {
      hvOff.hdr2 = 0L;
      hvOff.hdr2len = 0L;
    }
  hvOff.nsamps = nsamps;
  hvOff.nlines = nlines;


/******************
 *  VH Sub-Image  *
 ******************/

  vhOff.img = IMAGEoff2;
  vhOff.hdr1 = nbytes0 + nbytes1;

  if (hdr2len2 > 0)		/* A Header-2 exists for VH sub-image */
    {
      vhOff.hdr2 = nbytes0 + nbytes1 + rl2;
      vhOff.hdr2len = hdr2len2;
    }
  else
    /* No header-2... */
    {
      vhOff.hdr2 = 0L;
      vhOff.hdr2len = 0L;
    }
  vhOff.nsamps = nsamps;
  vhOff.nlines = nlines;


/******************
 *  VV Sub-Image  *
 ******************/

  vvOff.img = IMAGEoff3;
  vvOff.hdr1 = nbytes0 + nbytes1 + nbytes2;

  if (hdr2len3 > 0)		/* A Header-2 exists for VH sub-image */
    {
      vvOff.hdr2 = nbytes0 + nbytes1 + nbytes2 + rl3;
      vvOff.hdr2len = hdr2len3;
    }
  else
    /* No header-2... */
    {
      vvOff.hdr2 = 0L;
      vvOff.hdr2len = 0L;
    }
  vvOff.nsamps = nsamps;
  vvOff.nlines = nlines;

  return 0;

}				/* end of getImgLoc() */
