/* =========================================================================
 * =========================================================================
 * == [DISCLAIMER]: THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS   ==
 * == RELEASED "AS IS".  THE U.S. GOVERNMENT MAKES NO WARRANTY OF ANY     ==
 * == KIND, EXPRESS OR IMPLIED, CONCERNING THIS SOFTWARE AND ANY          ==
 * == ACCOMPANYING DOCUMENTATION, INCLUDING, WITHOUT LIMITATION, ANY      ==
 * == WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  ==
 * == IN NO EVENT WILL THE U.S. GOVERNMENT BE LIABLE FOR ANY DAMAGES      ==
 * == ARISING OUT OF THE USE, OR INABILITY TO USE, THIS SOFTWARE OR ANY   ==
 * == ACCOMPANYING DOCUMENTATION, EVEN IF INFORMED IN ADVANCE OF THE      ==
 * == POSSIBLITY OF SUCH DAMAGES.                                         ==
 * =========================================================================
 * ========================================================================= */

/*-------------------------------------------------------------------------
 *                        Routine: js_ddb2jpeg (Version 1.0)
 *                         Author: John F. Querns, Veridian Inc., Veda Ops
 *                           Date: 28 May 1998
 *
 *-------------------------------------------------------------------------
 *
 * Purpose:  This routine converts Jumpstart DDB complex imagery with Phoenix
 *           headers to 8-bit JPEG format.  It uses a "tiling" method to
 *           read in and process data. This greatly speeds up the processing of
 *           large data sets. This routine provides the following features:
 *
 *           1) Can optionally dump Phoenix header to output file.  Filename
 *              is automatically generated.
 *
 *           2) Can convert whole input image to 8-bit JPEG <or>
 *
 *           3) Can optionally convert only a specific portion (or tile)
 *              of data to 8-bit.  Program currently defaults to 8 tiles
 *              (numbered 0-7) for input images whose height is greater
 *              than 1024.
 *
 *           4) Can automatically enhance output image..
 *
 *-------------------------------------------------------------------------
 *
 * [Calls]:
 *
 *     char *read_switch()   -- Utility to extract arguments from
 *                              input switches.  This was extracted
 *                              from code developed by Otto Milvang,
 *                              U. of Oslo and is part of the B-LAB
 *                              XITE software.
 *
 *                              Copyright 1990, Blab, UiO
 *                              Image processing lab,
 *                              Department of Informatics
 *                              University of Oslo
 *
 *------------------------------------------------------------------------
 *
 * [Methodology]:
 *
 *     This routine will use a new method to read in and process input
 *     data: tiling. Tile size follows the following scheme:
 *
 *           tile width  = image width
 *           tile height = image height     (image height < 1024)
 *                         image height / 8 (>= 1024)
 *
 *     The Phoenix header, as before, may be optionally written to an external
 *     output file. This routine WILL NOT HAVE a verbose/quiet switch.  Instead,
 *     information messages will be automatic.  Also, this routine does not
 *     include auto-enhancement and auto-byteswapping.
 *
 *-------------------------------------------------------------------------
 *
 *     Complex to Magnitude method:
 *     ---------------------------
 *
 *          MAGPixel = Sqroot( RealPix**2 + ImagPix**2 )
 *
 *
 *     8-bit Scaling Method (on magnitude of complex data):
 *     ---------------------------------------------------
 *
 *          Given: OP     = Output pixel
 *                 IP     = Input pixel
 *                 RANGE  = Range of gray values in input image (MAX-MIN)
 *                 MIN    = Minimum pixel value in input image
 *                 T      = Temporary value
 *
 *          Then:  T   =   ((FLOAT) (IP - MIN) / RANGE) * 255.0
 *                 OP  = (unsigned char) (T + 0.5);
 *
 *------------------------------------------------------------------------
 *
 * [ASCII Header]:
 *
 *    This routine includes a switch option (-h) that allows the dumping
 *    of the Phoenix header to an output file.  The output header
 *    will use name of input file as its basename and will have the
 *    extension: ".hdr".  The default mode is NOT to dump header.
 *
 *------------------------------------------------------------------------
 *
 * [Syntax/Usage]:
 *
 *    js_ddb2jpeg -i < DDB file input> -o <JPEG output> [-h] -e -t [tile]
 *
 *            Options: -h       -- Dump Phoenix header
 *                     -e       -- Auto-enhance output image
 *                     -t [0-7] -- Output file contains data from a
 *                                 specific tile [0 through 7]
 *
 *-----------------------------------------------------------------------
 *
 * [Contacts]:
 *
 *   John F. Querns
 *   Veridian Inc., Veda Operations (Dayton Group)
 *   5200 Springfield Pike, Dayton, OH 45431
 *   email: jquerns@dytn.veridian.com
 *
 *   Veridian/Veda Contractor Area
 *   Area B  Bldg 23  Rm 115
 *   2010 Fifth Street
 *   Wright Patterson AFB, OH  45433-7001
 *   Work : (937) 255-1116, Ext 2818
 *   email: jquerns@mbvlab.wpafb.af.mil
 *
 *-------------------------------------------------------------------------
 *
 * [Versions]:
 *
 *   1.0: 28 May 1998
 *
 *------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <math.h>
#include "jpeglib.h"

#ifndef FALSE
#define FALSE  0
#endif

#ifndef TRUE
#define TRUE   1
#endif

#define MAX(x,y)   (((x) > (y)) ? (x) : (y))
#define MIN(x,y)   (((x) < (y)) ? (x) : (y))


struct cmplxnum
{
  float           r;
  float           i;
};

extern char    *read_switch();


main(argc, argv)

  int             argc;
  char           *argv[];

{
  FILE           *DDBfp = NULL;	/* Ptr to input DDB file....      */
  FILE           *JPGfp = NULL;	/* Ptr to output JPEG file..      */
  FILE           *HDRfp = NULL;	/* Ptr to opt output header file.. */

  int             i, j, w, h, h2, n;
  int             numtiles, npix;

  char           *DDBname = NULL;	/* Ptr to input DDB filename        */
  char           *JPGname = NULL;	/* Ptr to output JPEG filename      */
  char            HDRname[80];	/* Phoenix header filename..        */

  int             HDRflag = FALSE;	/* If TRUE, dump Phoenix header     */
  char           *HDRopt;

  char           *ENHopt;
  int             ENHflag;	/* If TRUE, auto-enhance output     */

  char           *QFopt;
  int             qfactor;
  int             region1, region2;
  int             region3, region4;

  int             TILEflag;	/* If TRUE, output=specific tile    */
  char           *TILEopt;
  long            TILEval;
  long            TILEstart, TILEend;

  char           *comment = "JumpStart DDB Complex to 8-Bit TIFF..";

  long            phlen, nhlen, phsigsize, numgot;
  long            magloc, imgdatasize;
  long            tile_wid, tile_hgt, tile_size;
  long            fullTileSize, partialTileSize;
  long            numFullTilePixels, numPartTilePixels;
  float           percentEachRead, percentComplete;
  char           *tptr = NULL, *phdr;
  unsigned char   tbuff[2048], ival;

  long            lmax, lmin, lrange;
  double          d1, dreal, dimag;
  float           fscale, f1, f2, fmax, fmin, frange, magval;

  float          *MAINbuf;	/* Main processing buffer  */
  unsigned char  *TILEbuf;	/* Buffer to hold each TILE */
  struct cmplxnum *C_TILEbuf;	/* Ptr to Complex I-Q data */
  unsigned char  *pic8;

  /* Enhancement Algorithm Variables */
  int             hist8[256];
  int             histmax, histmin, histrange, numcolors = 0;
  int             maxPixelCountBin, minPixelCountBin;
  int             CountBinRange;
  float           fsum, t1, hval;
  float           scaleAdj, threshVal, threshLim;

  /**********************************************************
   *                   JPEG-RELATED VARIABLES               *
   **********************************************************/

  /* This struct contains the JPEG compression parameters and pointers to
   * working space (which is allocated as needed by the JPEG library). It is
   * possible to have several such structures, representing multiple
   * compression/decompression processes, in existence at once.  We refer to
   * any one struct (and its associated working data) as a "JPEG object". */
  struct jpeg_compress_struct cinfo;

  /* This struct represents a JPEG error handler.  It is declared separately
   * because applications often want to supply a specialized error handler
   * (see the second half of this file for an example).  But here we just
   * take the easy way out and use the standard error handler, which will
   * print a message on stderr and call exit() if compression fails. */
  struct jpeg_error_mgr jerr;

  JSAMPROW        rowptr[1];
  unsigned int    clen;




  /************************ B E G I N  C O D E ****************************/

  if (argc < 2)
  {
    fprintf(stderr,
	    "\n\nUsage: js_ddb2jpeg -i <DDB File> -o <JPEG File> -e -h -q [qf] -t [0-7]");
    fprintf(stderr, "\n  Options: -e       =  Contrast Enhancement Factor [Def: none]");
    fprintf(stderr, "\n           -h       =  Dump Phoenix (ASCII) header [Def: none]");
    fprintf(stderr, "\n           -q [qf]  =  Compression  Quality Factor [5-95, Def:75]");
    fprintf(stderr, "\n           -t [0-7] =  Output data from specific tile \n\n");
    exit(1);
  }
  /* Extract Input & Output filenames */
  DDBname = read_switch(&argc, argv, "-i", 1, NULL);
  JPGname = read_switch(&argc, argv, "-o", 1, NULL);

  /* Check for auto-enhancement option */
  ENHopt = read_switch(&argc, argv, "-e", 0, NULL);
  if (ENHopt != (char *) NULL)
  {
    ENHflag = TRUE;
  } else
  {
    ENHflag = FALSE;
  }

  /****************************************************
   * Compression Quality Factor:                      *
   * --------------------------                       *
   *                                                  *
   * 1) If "-q" NOT ENTERED, routine ASSUMES quality  *
   *    factor of 75.                                 *
   *                                                  *
   * 2) If "-q" ENTERED, routine assigns inputted     *
   *    quality factor as long as it meets limits.    *
   *                                                  *
   ****************************************************/
  QFopt = read_switch(&argc, argv, "-q", 1, NULL);
  if (QFopt == (char *) NULL)
  {
    qfactor = 75;
  } else
  {
    qfactor = atoi(QFopt);
    if ((qfactor < 5) ||
	(qfactor > 95))
    {
      fprintf(stderr, "\n\nError: compression quality factor range is [5-95]\n\n");
      exit(1);
    }
  }

  /* Check for specific TILE extraction */
  TILEopt = read_switch(&argc, argv, "-t", 1, NULL);
  if (TILEopt == (char *) NULL)
  {
    /* Extract ALL data: default */
    TILEflag = FALSE;
    TILEval = 0;
  } else
  {
    TILEflag = TRUE;
    TILEval = atol(TILEopt);

    /* Check TILE value */
    if ((TILEval < 0) ||
	(TILEval > 7))
    {
      fprintf(stderr,
	      "Error: tile value can only be 0 to 7!\n\n");
      exit(1);
    }
  }

  /* If TILE flag is TRUE, append TILE value to output name */
  if (TILEflag == TRUE)
  {
    strcat(JPGname, TILEopt);
  }
  /* Extract optional arguments (if any) */
  HDRopt = read_switch(&argc, argv, "-h", 0, NULL);
  if (HDRopt != (char *) NULL)
  {
    HDRflag = TRUE;
  }
  /* Form output header filename */
  if (HDRflag == TRUE)
  {
    HDRname[0] = '\0';
    strcat(HDRname, DDBname);
    strcat(HDRname, ".hdr");
  }
  printf("\n\nJumpStart DDB to JPEG conversion: started!\n");


  /***************** PHOENIX HEADER FILE PROCESSING AREA ********************/

  DDBfp = fopen(DDBname, "rb");
  if (DDBfp == NULL)
  {
    fprintf(stderr,
	    "\n\nError: Unable to open [%s] for reading!\n\n", DDBname);
    exit(1);
  }
  /****************************************************
   * Read first 2K bytes to figure out some header    *
   * parameters....                                   *
   ****************************************************/

  printf("Determining input DDB image/hdr information...\n");
  n = fread(tbuff, sizeof(char), 2048, DDBfp);
  if (n != 2048)
  {
    fprintf(stderr,
	    "\nError: only able to read [%d] of first 2048 bytes!\n\n",
	    n);
    fclose(DDBfp);
    exit(1);
  }
  rewind(DDBfp);

  /* Extract Phoenix Summary header length */
  tptr = (char *) strstr(tbuff, "PhoenixHeaderLength= ");
  if (tptr == (char *) NULL)
  {
    fprintf(stderr, "Can not find PhoenixHeaderLength field!\n");
    fclose(DDBfp);
    exit(1);
  } else
  {
    sscanf((tptr + 20), "%d", &phlen);
  }

  /* Extract Phoenix Signal Size */
  tptr = (char *) strstr(tbuff, "PhoenixSigSize= ");
  if (tptr == (char *) NULL)
  {
    fprintf(stderr, "Can not find PhoenixSigSize field!\n");
    fclose(DDBfp);
    exit(1);
  } else
  {
    sscanf((tptr + 15), "%d", &phsigsize);
  }

  /* Check for and extract native header length */
  tptr = (char *) strstr(tbuff, "native_header_length= ");
  if (tptr == (char *) NULL)
  {
    fprintf(stderr, "Can not find native header length field!\n");
    fclose(DDBfp);
    exit(1);
  } else
  {
    sscanf((tptr + 21), "%d", &nhlen);
  }

  /* Extract DDB file column width */
  tptr = (char *) strstr(tbuff, "NumberOfColumns= ");
  if (tptr == (char *) NULL)
  {
    fprintf(stderr,
	    "Error: Can not find DDB image width field");
    fclose(DDBfp);
    exit(1);
  } else
  {
    sscanf((tptr + 16), "%d", &w);
  }

  /* Extract DDB file row height */
  tptr = (char *) strstr(tbuff, "NumberOfRows= ");
  if (tptr == (char *) NULL)
  {
    fprintf(stderr,
	    "Error: Can not find DDB image height field!");
    fclose(DDBfp);
    exit(1);
  } else
  {
    sscanf((tptr + 13), "%d", &h);
  }

  /* Set up TILE Parameters */
  tile_wid = w;			/* Always set to width of image row */

  if (h < 1024)
  {
    tile_hgt = h;
  } else
  {
    tile_hgt = (long) (h / 8);
  }

  tile_size = tile_wid * tile_hgt * 8;

  /* Set up number of pixels */
  switch (TILEflag)
  {
  case FALSE:
    npix = w * h;
    break;
  case TRUE:
    npix = tile_wid * tile_hgt;
    break;
  }

  /*******************************************************
   * Allocate memory to header buffer, read Phoenix hdr, *
   * and write out to output file...                     *
   *******************************************************/
  if (HDRflag == TRUE)
  {
    HDRfp = fopen(HDRname, "w");
    if (HDRfp == NULL)
    {
      fprintf(stderr,
	      "Error: unable to open header file [%s] for writing!\n",
	      HDRname);
      fclose(DDBfp);
      exit(1);
    }
    phdr = (char *) malloc(phlen + 1);
    if (phdr == (char *) NULL)
    {
      fprintf(stderr,
	      "Error: unable to allocate header memory!\n");
      fclose(DDBfp);
      fclose(HDRfp);
      unlink(HDRname);
      exit(1);
    }
    fread(phdr, sizeof(char), phlen, DDBfp);
    fwrite(phdr, sizeof(char), phlen, HDRfp);
    fclose(HDRfp);
    free(phdr);
    printf("Phoenix header written to [%s]...\n", HDRname);

  }
  /********************** DDB IMAGE DATA PROCESSING AREA ********************/


  /***********************************************************
   * Calculate image data size...                            *
   * Calculate number of full and partial tiles to process   *
   * Calculate percent of image processed with each tile     *
   * Calculate partial tile datasize (pixels & bytes)        *
   * Display user info...                                    *
   ***********************************************************/
  /* Calculate Image Data Size */
  imgdatasize = phsigsize - phlen;

  /* Calculate number of full data tiles */
  numtiles = (int) (imgdatasize / tile_size);

  /* Calculate percent each tile read */
  f1 = (float) tile_size / (float) imgdatasize;
  percentEachRead = f1 * 100.0;

  /* Calculate Num of Full Tile pixels */
  numFullTilePixels = tile_wid * tile_hgt;

  /* Calculate partial tile datasize */
  partialTileSize = imgdatasize - (numtiles * tile_size);

  /* Calculate Num of Partial Tile pixels */
  numPartTilePixels = (long) (partialTileSize / 8);

  /* Display User Info */
  printf("\n\nJumpStart DDB Image Processing Information:\n\n");
  printf("  Input Image datatype           = [COMPLEX I-Q]\n");
  printf("  Input Image datasize (bytes)   = [%ld]\n", imgdatasize);
  printf("  Input Image width (columns)    = [%d]\n", w);
  printf("  Input Image height (rows)      = [%d]\n\n", h);
  printf("  Number of Tiles                = [%d]\n", numtiles);
  printf("  Tile width (columns)           = [%d]\n", tile_wid);
  printf("  Tile height (rows)             = [%d]\n", tile_hgt);
  if (TILEflag == FALSE)
  {
    printf("  Processing Tiles               = 0-7 plus any excess...\n\n");
  } else
  {
    printf("  Processing Tile                = %d\n\n", TILEval);
  }

  /* Allocate space for 32-bit MAIN buffer */

  MAINbuf = (float *) malloc(npix * sizeof(float));
  if (MAINbuf == (float *) NULL)
  {
    fprintf(stderr,
	    "Error: unable to allocate space for main buffer!\n");
    fclose(DDBfp);
    exit(1);
  }
  /* Allocate space for unsigned char TILE buffer */
  TILEbuf = (unsigned char *) malloc(tile_size);
  if (TILEbuf == (unsigned char *) NULL)
  {
    fprintf(stderr,
	    "Error: unable to allocate space for tile buffer!\n");
    fclose(DDBfp);
    free(MAINbuf);
    exit(1);
  }
  /*******************************************************************
   *                        PROCESSING LOOP                          *
   *******************************************************************
   * For each tile:                                                  *
   * -------------                                                   *
   *                                                                 *
   *  a) Get TILE of data from file..                                *
   *  b) Re-cast TILE buffer as COMPLEX...                           *
   *  c) Process each tile. For each complex pixel in tile buffer    *
   *     -- Convert to 32-bit float magnitude                        *
   *     -- Update max and min values...                             *
   *     -- Store in main (float) buffer...                          *
   *  d) Update percent read variables...let user know...            *
   *******************************************************************/

  /* Set up MAX & MIN variable */
  fmax = -100000.0;
  fmin = 100000.0;

  /* Determine TILE start & end */
  switch (TILEflag)
  {
  case FALSE:			/* Process ALL data tiles */
    TILEstart = 0;
    TILEend = numtiles;

    /* Point to beginning of DDB complex data.. */
    magloc = (long) phlen;
    fseek(DDBfp, magloc, 0);

    break;

  case TRUE:			/* Process 1 tile */
    TILEstart = TILEval;
    TILEend = TILEval + 1;

    /* Point to beginning of DDB complex TILE data.. */
    magloc = (long) phlen + (TILEstart * tile_size);
    fseek(DDBfp, magloc, 0);

    break;
  }

  /* Process FULL tiles first */
  printf("Performing Initial Processing:\n\n");

  for (i = TILEstart; i < TILEend; i++)
  {
    /* Get 1 Full TILE of data */
    numgot = fread(TILEbuf, sizeof(char), tile_size, DDBfp);

    /* Re-cast as COMPLEX */
    C_TILEbuf = (struct cmplxnum *) TILEbuf;

    /* Process each TILE */
    for (j = 0; j < numFullTilePixels; j++)
    {
      /* Convert pixel to float magnitude value */
      dreal = (double) (C_TILEbuf[j].r * C_TILEbuf[j].r);
      dimag = (double) (C_TILEbuf[j].i * C_TILEbuf[j].i);
      d1 = sqrt(dreal + dimag);
      magval = (float) d1;

      /* Update float max and min values */
      fmax = MAX(fmax, magval);
      fmin = MIN(fmin, magval);

      /* Store in main buffer */
      switch (TILEflag)
      {
      case FALSE:		/* Do ALL data */
	MAINbuf[j + (i * numFullTilePixels)] = magval;
	break;
      case TRUE:		/* Do 1 TILE   */
	MAINbuf[j] = magval;
	break;
      }
    }

    /* Calculate percent complete */
    if (TILEflag == FALSE)
    {
      percentComplete = percentEachRead + (i * percentEachRead);
      printf("  [%6.3f] percent completed...\n", percentComplete);
    }
  }

  /* Process (last) PARTIAL Tile (if TILEflag = FALSE) */

  if (TILEflag == FALSE)
  {
    numgot = fread(TILEbuf, sizeof(char), partialTileSize, DDBfp);
    if (numgot != partialTileSize)
    {
      fprintf(stderr,
	      "\nError: in reading remaining DDB data!\n\n");
      fclose(DDBfp);
      free(MAINbuf);
      free(TILEbuf);
      C_TILEbuf = NULL;
      exit(1);
    }
    /* Re-cast as COMPLEX */
    C_TILEbuf = (struct cmplxnum *) TILEbuf;

    /* Convert partial tile data */
    for (j = 0; j < numPartTilePixels; j++)
    {
      /* Convert pixel to float magnitude value */
      dreal = (double) (C_TILEbuf[j].r * C_TILEbuf[j].r);
      dimag = (double) (C_TILEbuf[j].i * C_TILEbuf[j].i);
      d1 = sqrt(dreal + dimag);
      magval = (float) d1;

      /* Update float max and min values */
      fmax = MAX(fmax, magval);
      fmin = MIN(fmin, magval);

      /* Store in main buffer */
      MAINbuf[j + (numFullTilePixels * numtiles)] = magval;
    }
  }
  /* Close input file..free memory..NULL pointers */
  fclose(DDBfp);
  free(TILEbuf);
  C_TILEbuf = NULL;

  printf("\nInitial processing complete...\n\n");


  /******************** 8-BIT SCALING AREA **************************/

  printf("Scaling JumpStart DDB image data to 8-bit...\n");

  /* Allocate 8-bit buffer */
  pic8 = (unsigned char *) calloc((size_t) npix, (size_t) 1);
  if (pic8 == (unsigned char *) NULL)
  {
    fprintf(stderr,
	    "Error: Unable to malloc output 8-bit memory!\n");
    free(MAINbuf);
    exit(1);
  }
  /* Do 8-bit linear scaling of MAINbuf data */
  frange = fmax - fmin;
  fscale = 255.0 / frange;

  for (i = 0; i < npix; i++)
  {
    f1 = (float) MAINbuf[i] - fmin;
    f2 = f1 * fscale;
    pic8[i] = (unsigned char) (f2 + 0.5);
  }
  free(MAINbuf);


  /************* CONTRAST ENHANCEMENT AREA ********************/

  if (ENHflag == TRUE)
  {
    printf("Contrast-Adjusting 8-bit JumpStart DDB output data...\n");

    /* Initialize histogram array */
    for (i = 0; i < 256; i++)
    {
      hist8[i] = 0;
    }

    /* Calculate histogram of 8-bit JumpStart DDB data */
    for (i = 0; i < npix; i++)
    {
      ival = pic8[i];
      hist8[ival]++;
    }

    /******************************************
     * Calculate number of grayscale values.. *
     ******************************************/
    for (i = 0; i < 256; i++)
    {
      if (hist8[i] > 0)
      {
	numcolors++;
      }
    }

    /* Find MAX & MIN COUNTS of Histogram BINS */
    histmax = 0;
    histmin = npix;
    for (i = 0; i < 256; i++)
    {
      histmax = MAX(hist8[i], histmax);
      histmin = MIN(hist8[i], histmin);
    }

    /* Find BIN where MAX & MIN Pixel Counts are */
    maxPixelCountBin = 0;	/* Assume Bin#0   has maximum pixel count.. */
    minPixelCountBin = 255;	/* Assume Bin#255 has minimum pixel count.. */
    for (i = 0; i < 256; i++)
    {
      if (hist8[i] == histmax)
      {				/* Do this just in case > 1 bin has histmax
				 * values */
	maxPixelCountBin = MAX(maxPixelCountBin, i);
      }
      if (hist8[i] == histmin)
      {				/* Do this just in case > 1 bin has histmin
				 * values */
	minPixelCountBin = MIN(minPixelCountBin, i);
      }
    }
    CountBinRange = abs(maxPixelCountBin - minPixelCountBin);

    /* Adjust Pixel Contrast */
    for (i = 0; i < npix; i++)
    {
      ival = pic8[i];
      if (minPixelCountBin > maxPixelCountBin)
      {
	threshVal = (float) (minPixelCountBin - maxPixelCountBin);
	if (ival > threshVal)
	{
	  f1 = 255.0;
	} else
	{
	  scaleAdj = 255.0 / threshVal;
	  f1 = scaleAdj * (float) ival;
	}
      } else
      {
	f1 = 3.0 * (float) ival;
      }

      if (f1 > 255.0)
      {
	pic8[i] = 255;
      } else
      {
	pic8[i] = (unsigned char) (f1 + 0.5);
      }
    }

  }
  /************* MAIN JPEG PROCESSING AREA ********************/

  printf("Setting up JPEG parameters....\n");

  /* Step 1: allocate and initialize JPEG compression object */

  /* We have to set up the error handler first, in case the initialization
   * step fails.  (Unlikely, but it could happen if you are out of memory.)
   * This routine fills in the contents of struct jerr, and returns jerr's
   * address which we place into the link field in cinfo. */
  cinfo.err = jpeg_std_error(&jerr);

  /* Now we can initialize the JPEG compression object. */
  jpeg_create_compress(&cinfo);

  /* Step 2: specify data destination (eg, a file) */
  JPGfp = fopen(JPGname, "wb");
  if (JPGfp == NULL)
  {
    fprintf(stderr,
	    "Error: unable to open output JPEG file [%s]\n",
	    JPGname);
    free(pic8);
    exit(1);
  }
  jpeg_stdio_dest(&cinfo, JPGfp);

  /* Step 3: Set parameters for compression..these 4 parameters MUST be
   * filled in... */
  switch (TILEflag)
  {
  case FALSE:			/* Using ALL data */
    cinfo.image_width = w;
    cinfo.image_height = h;
    cinfo.input_components = 1;
    cinfo.in_color_space = JCS_GRAYSCALE;
    break;

  case TRUE:			/* Using just 1 TILE of data */
    cinfo.image_width = tile_wid;
    cinfo.image_height = tile_hgt;
    cinfo.input_components = 1;
    cinfo.in_color_space = JCS_GRAYSCALE;
    break;
  }

  /* Now use the library's routine to set default compression parameters.
   * (You must set at least cinfo.in_color_space before calling this, since
   * the defaults depend on the source color space.) */
  jpeg_set_defaults(&cinfo);

  /* Change parameters you want to change..limit to baseline-JPEG values */
  /* Set quality...no smoothing */
  jpeg_set_quality(&cinfo, qfactor, TRUE);
  cinfo.smoothing_factor = 0;

  printf("\nCompressing data...");

  /* Step 4: Start compressor TRUE ensures we write a complete
   * interchange-JPEG file */
  jpeg_start_compress(&cinfo, TRUE);

  /* Step 5: while (scan lines remain to be written)....
   * jpeg_write_scanlines(...);
   * 
   * Here we use the library's state variable cinfo.next_scanline as the loop
   * counter, so that we don't have to keep track ourselves. To keep things
   * simple, we pass one scanline per call; you can pass more if you wish,
   * though. */

  printf("Writing compressed data to JPEG file: [%s]\n", JPGname);

  switch (TILEflag)
  {
  case FALSE:			/* All data */
    while (cinfo.next_scanline < cinfo.image_height)
    {
      rowptr[0] = (JSAMPROW) & pic8[cinfo.next_scanline * w];
      (void) jpeg_write_scanlines(&cinfo, rowptr, (JDIMENSION) 1);
    }
    break;

  case TRUE:			/* 1 TILE of data */
    while (cinfo.next_scanline < cinfo.image_height)
    {
      rowptr[0] = (JSAMPROW) & pic8[cinfo.next_scanline * tile_wid];
      (void) jpeg_write_scanlines(&cinfo, rowptr, (JDIMENSION) 1);
    }
    break;
  }

  /* Step 6: Finish compression */
  jpeg_finish_compress(&cinfo);

  /* After finish_compress, we can close the output file. */
  fclose(JPGfp);

  /* Step 7: release JPEG compression object */
  jpeg_destroy_compress(&cinfo);


  /* Free space */
  free(pic8);

  printf("\nJumpStart DDB to JPEG conversion: completed!\n\n");

  exit(0);
}
