/* =========================================================================
 * =========================================================================
 * == [DISCLAIMER]: THIS SOFTWARE AND ANY ACCOMPANYING DOCUMENTATION IS   ==
 * == RELEASED "AS IS".  THE U.S. GOVERNMENT MAKES NO WARRANTY OF ANY     ==
 * == KIND, EXPRESS OR IMPLIED, CONCERNING THIS SOFTWARE AND ANY          ==
 * == ACCOMPANYING DOCUMENTATION, INCLUDING, WITHOUT LIMITATION, ANY      ==
 * == WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  ==
 * == IN NO EVENT WILL THE U.S. GOVERNMENT BE LIABLE FOR ANY DAMAGES      ==
 * == ARISING OUT OF THE USE, OR INABILITY TO USE, THIS SOFTWARE OR ANY   ==
 * == ACCOMPANYING DOCUMENTATION, EVEN IF INFORMED IN ADVANCE OF THE      ==
 * == POSSIBLITY OF SUCH DAMAGES.                                         ==
 * =========================================================================
 * ========================================================================= */

/*-------------------------------------------------------------------------
 *                        Routine: js_ddb2raw (Version 1.0)
 *                         Author: John F. Querns, Veridian Inc, Veda Ops
 *                           Date: 28 May 1998
 *
 *-------------------------------------------------------------------------
 *
 * Purpose:  This routine converts Jumpstart DDB complex imagery with Phoenix
 *           headers to a RAW binary format + an ASCII Phoenix header.  It
 *           uses a "tiling" method to read in and process data. This greatly
 *           speeds up the processing of large data sets. This routine provides
 *           the following features:
 *
 *           1) Auto-dumps Phoenix header to output file.  Filename is
 *               automatically generated.
 *
 *           2) Convert whole or partial input image to RAW binary
 *
 *           3) Output type can be original complex, 32-bit float magnitude,
 *              or 8-bit magnitude
 *
 *-------------------------------------------------------------------------
 *
 * [Calls]:
 *
 *     char *read_switch()   -- Utility to extract arguments from
 *                              input switches.  This was extracted
 *                              from code developed by Otto Milvang,
 *                              U. of Oslo and is part of the B-LAB
 *                              XITE software.
 *
 *                              Copyright 1990, Blab, UiO
 *                              Image processing lab,
 *                              Department of Informatics
 *                              University of Oslo
 *
 *------------------------------------------------------------------------
 *
 * [Methodology]:
 *
 *     This routine will use a new method to read in and process input
 *     data: tiling. Tile size follows the following scheme:
 *
 *           tile width  = image width
 *           tile height = image height     (image height < 1024)
 *                         image height / 8 (>= 1024)
 *
 *-------------------------------------------------------------------------
 *
 *     Complex to Magnitude method:
 *     ---------------------------
 *
 *          MAGPixel = Sqroot( RealPix**2 + ImagPix**2 )
 *
 *
 *     8-bit Scaling Method (on magnitude of complex data):
 *     ---------------------------------------------------
 *
 *          Given: OP     = Output pixel
 *                 IP     = Input pixel
 *                 RANGE  = Range of gray values in input image (MAX-MIN)
 *                 MIN    = Minimum pixel value in input image
 *                 T      = Temporary value
 *
 *          Then:  T   =   ((FLOAT) (IP - MIN) / RANGE) * 255.0
 *                 OP  = (unsigned char) (T + 0.5);
 *
 *------------------------------------------------------------------------
 *
 * [ASCII Header]:
 *
 *    This routine automatically dumps the Phoenix header to an output file.
 *    the output header will use name of input file as its basename and will
 *    have the extension: ".hdr".  The default mode is NOT to dump header.
 *
 *------------------------------------------------------------------------
 *
 * [Syntax/Usage]:
 *
 *    js_ddb2raw -i < DDB file input> -o <RAW output> d [0-2] -t [tile]
 *
 *            Options: -d [0-2] -- Output datatype: 0=complex, 1=mag32, 2=mag8 [Def:0]
 *                     -t [0-7] -- Output file contains data from a
 *                                 specific tile [0 through 7]
 *
 *-----------------------------------------------------------------------
 *
 * [Contacts]:
 *
 *   John F. Querns
 *   Veridian Inc., Veda Operations (Dayton Group)
 *   5200 Springfield Pike, Dayton, OH 45431
 *   email: jquerns@dytn.veridian.com
 *
 *   Veridian/Veda Contractor Area
 *   Area B  Bldg 23  Rm 115
 *   2010 Fifth Street
 *   Wright Patterson AFB, OH  45433-7001
 *   Work : (937) 255-1116, Ext 2818
 *   email: jquerns@mbvlab.wpafb.af.mil
 *
 *-------------------------------------------------------------------------
 *
 * [Versions]:
 *
 *   1.0: 28 May 1998
 *
 *------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <math.h>

#ifndef FALSE
#define FALSE  0
#endif

#ifndef TRUE
#define TRUE   1
#endif

#define MAX(x,y)   (((x) > (y)) ? (x) : (y))
#define MIN(x,y)   (((x) < (y)) ? (x) : (y))

#define ORIGTYPE   0
#define MAG32TYPE  1
#define MAG8TYPE   2

struct cmplxnum
{
  float           r;
  float           i;
};

extern char    *read_switch();


main(argc, argv)

  int             argc;
  char           *argv[];

{
  FILE           *DDBfp = NULL;	/* Ptr to input DDB file....      */
  FILE           *RAWfp = NULL;	/* Ptr to output RAW file..       */
  FILE           *HDRfp = NULL;	/* Ptr to opt output header file.. */

  int             i, j, w, h, n;
  int             numtiles, npix;

  char           *DDBname = NULL;	/* Ptr to input DDB filename        */
  char           *RAWname = NULL;	/* Ptr to output RAW filename       */
  char            HDRname[80];	/* Phoenix header filename..        */

  char           *DATAopt;
  int             DATAval;

  int             TILEflag;	/* If TRUE, output=specific tile    */
  char           *TILEopt;
  long            TILEval;
  long            TILEstart, TILEend;

  long            phlen, nhlen, phsigsize, numgot, numwrote;
  long            magloc, imgdatasize;
  long            tile_wid, tile_hgt, tile_size;
  long            fullTileSize, partialTileSize;
  long            numFullTilePixels, numPartTilePixels;
  float           percentEachRead, percentComplete;
  char           *tptr = NULL, *phdr;
  unsigned char   tbuff[2048], ival;

  long            lmax, lmin, lrange;
  double          d1, dreal, dimag;
  float           fscale, f1, f2, fmax, fmin, frange, magval;

  float          *MAINbuf;	/* Main processing buffer  */
  unsigned char  *TILEbuf;	/* Buffer to hold each TILE */
  struct cmplxnum *C_TILEbuf;	/* Ptr to Complex I-Q data */
  unsigned char  *pic8;		/* Ptr to 8-bit buffer     */


  /************************ B E G I N  C O D E ****************************/

  if (argc < 2)
  {
    fprintf(stderr,
     "\n\nUsage: js_ddb2raw -i <DDB File> -o <RAW File> -d [0-2] -t [0-7]");
    fprintf(stderr, "\nOptions: -d [0-2] = Output datatype, 0=complex 1=mag32, 2=mag8 [Def:1]");
    fprintf(stderr, "\n         -t [0-7] = Output data from specific tile \n\n");
    exit(1);
  }
  /* Extract Input & Output filenames */
  DDBname = read_switch(&argc, argv, "-i", 1, NULL);
  RAWname = read_switch(&argc, argv, "-o", 1, NULL);


  /* Check for output data type */
  DATAopt = read_switch(&argc, argv, "-d", 1, NULL);
  if (DATAopt == (char *) NULL)
  {
    /* Option not selected..dump RAW data as original COMPLEX: default */
    DATAval = MAG32TYPE;
  } else
  {				/* Option selected..check for validity */
    DATAval = atoi(DATAopt);

    /* Check output data value */
    if ((DATAval < 0) ||
	(DATAval > 2))
    {
      fprintf(stderr,
      "Error: output data value can only be complex,  mag32, or mag8!\n\n");
      exit(1);
    }
  }


  /* Check for specific TILE extraction */
  TILEopt = read_switch(&argc, argv, "-t", 1, NULL);
  if (TILEopt == (char *) NULL)
  {
    /* Extract ALL data: default */
    TILEflag = FALSE;
    TILEval = 0;
  } else
  {
    TILEflag = TRUE;
    TILEval = atol(TILEopt);

    /* Check TILE value */
    if ((TILEval < 0) ||
	(TILEval > 7))
    {
      fprintf(stderr,
	      "Error: tile value can only be 0 to 7!\n\n");
      exit(1);
    }
  }

  /* If TILE flag is TRUE, append TILE value to output name */
  if (TILEflag == TRUE)
  {
    strcat(RAWname, TILEopt);
  }
  /* Form output header filename */
  HDRname[0] = '\0';
  strcat(HDRname, DDBname);
  strcat(HDRname, ".hdr");

  printf("\n\nJumpStart DDB to RAW Binary conversion: started!\n");


  /***************** PHOENIX HEADER FILE PROCESSING AREA ********************/

  DDBfp = fopen(DDBname, "rb");
  if (DDBfp == NULL)
  {
    fprintf(stderr,
	    "\n\nError: Unable to open [%s] for reading!\n\n", DDBname);
    exit(1);
  }
  /****************************************************
   * Read first 2K bytes to figure out some header    *
   * parameters....                                   *
   ****************************************************/

  printf("Determining input DDB image/hdr information...\n");
  n = fread(tbuff, sizeof(char), 2048, DDBfp);
  if (n != 2048)
  {
    fprintf(stderr,
	    "\nError: only able to read [%d] of first 2048 bytes!\n\n",
	    n);
    fclose(DDBfp);
    exit(1);
  }
  rewind(DDBfp);

  /* Extract Phoenix Summary header length */
  tptr = (char *) strstr(tbuff, "PhoenixHeaderLength= ");
  if (tptr == (char *) NULL)
  {
    fprintf(stderr, "Can not find PhoenixHeaderLength field!\n");
    fclose(DDBfp);
    exit(1);
  } else
  {
    sscanf((tptr + 20), "%d", &phlen);
  }

  /* Extract Phoenix Signal Size */
  tptr = (char *) strstr(tbuff, "PhoenixSigSize= ");
  if (tptr == (char *) NULL)
  {
    fprintf(stderr, "Can not find PhoenixSigSize field!\n");
    fclose(DDBfp);
    exit(1);
  } else
  {
    sscanf((tptr + 15), "%d", &phsigsize);
  }

  /* Check for and extract native header length */
  tptr = (char *) strstr(tbuff, "native_header_length= ");
  if (tptr == (char *) NULL)
  {
    fprintf(stderr, "Can not find native header length field!\n");
    fclose(DDBfp);
    exit(1);
  } else
  {
    sscanf((tptr + 21), "%d", &nhlen);
  }

  /* Extract DDB file column width */
  tptr = (char *) strstr(tbuff, "NumberOfColumns= ");
  if (tptr == (char *) NULL)
  {
    fprintf(stderr,
	    "Error: Can not find DDB image width field");
    fclose(DDBfp);
    exit(1);
  } else
  {
    sscanf((tptr + 16), "%d", &w);
  }

  /* Extract DDB file row height */
  tptr = (char *) strstr(tbuff, "NumberOfRows= ");
  if (tptr == (char *) NULL)
  {
    fprintf(stderr,
	    "Error: Can not find DDB image height field!");
    fclose(DDBfp);
    exit(1);
  } else
  {
    sscanf((tptr + 13), "%d", &h);
  }

  /* Set up TILE Parameters */
  tile_wid = w;			/* Always set to width of image row */

  if (h < 1024)
  {
    tile_hgt = h;
  } else
  {
    tile_hgt = (long) (h / 8);
  }

  tile_size = tile_wid * tile_hgt * 8;

  /* Set up number of pixels */
  switch (TILEflag)
  {
  case FALSE:
    npix = w * h;
    break;
  case TRUE:
    npix = tile_wid * tile_hgt;
    break;
  }

  /*******************************************************
   * Allocate memory to header buffer, read Phoenix hdr, *
   * and write out to output file...                     *
   *******************************************************/
  HDRfp = fopen(HDRname, "w");
  if (HDRfp == NULL)
  {
    fprintf(stderr,
	    "Error: unable to open header file [%s] for writing!\n",
	    HDRname);
    fclose(DDBfp);
    exit(1);
  }
  phdr = (char *) malloc(phlen + 1);
  if (phdr == (char *) NULL)
  {
    fprintf(stderr, "Error: unable to allocate header memory!\n");
    fclose(DDBfp);
    fclose(HDRfp);
    unlink(HDRname);
    exit(1);
  }
  fread(phdr, sizeof(char), phlen, DDBfp);
  fwrite(phdr, sizeof(char), phlen, HDRfp);
  fclose(HDRfp);
  free(phdr);

  printf("Phoenix header written to [%s]...\n", HDRname);


  /***************** DDB IMAGE DATA PROCESSING AREA ***************/


  /***********************************************************
   * Calculate image data size...                            *
   * Calculate number of full and partial tiles to process   *
   * Calculate percent of image processed with each tile     *
   * Calculate partial tile datasize (pixels & bytes)        *
   * Display user info...                                    *
   ***********************************************************/
  /* Calculate Image Data Size */
  imgdatasize = phsigsize - phlen;

  /* Calculate number of full data tiles */
  numtiles = (int) (imgdatasize / tile_size);

  /* Calculate percent each tile read */
  f1 = (float) tile_size / (float) imgdatasize;
  percentEachRead = f1 * 100.0;

  /* Calculate Num of Full Tile pixels */
  numFullTilePixels = tile_wid * tile_hgt;

  /* Calculate partial tile datasize */
  partialTileSize = imgdatasize - (numtiles * tile_size);

  /* Calculate Num of Partial Tile pixels */
  numPartTilePixels = (long) (partialTileSize / 8);

  /* Display User Info */
  printf("\n\nJumpStart DDB Image Processing Information:\n\n");
  printf("  Input Image datatype           = [COMPLEX I-Q]\n");
  printf("  Input Image datasize (bytes)   = [%ld]\n", imgdatasize);
  printf("  Input Image width (columns)    = [%d]\n", w);
  printf("  Input Image height (rows)      = [%d]\n\n", h);
  printf("  Number of Tiles                = [%d]\n", numtiles);
  printf("  Tile width (columns)           = [%d]\n", tile_wid);
  printf("  Tile height (rows)             = [%d]\n", tile_hgt);
  if (TILEflag == FALSE)
  {
    printf("  Processing Tiles               = 0-7 plus any excess...\n\n");
  } else
  {
    printf("  Processing Tile                = %d\n\n", TILEval);
  }


  /* Open output RAW file for writing */
  RAWfp = fopen(RAWname, "wb");
  if (RAWfp == NULL)
  {
    fprintf(stderr,
	    "Error: unable to open [%s] for writing!\n\n",
	    RAWname);
    exit(1);
  }
  /* Allocate space for 32-bit MAIN buffer if MAG Output DataType */
  switch (DATAval)
  {
  case MAG8TYPE:		/* 8-bit output        */
  case MAG32TYPE:		/* 32-bit float output */
    MAINbuf = (float *) malloc(npix * sizeof(float));
    if (MAINbuf == (float *) NULL)
    {
      fprintf(stderr,
	      "Error: unable to allocate space for main buffer!\n");
      fclose(DDBfp);
      fclose(RAWfp);
      unlink(RAWname);
      exit(1);
    }
    break;
  case ORIGTYPE:		/* Complex output      */
    break;
  }

  /* Allocate space for unsigned char TILE buffer */
  TILEbuf = (unsigned char *) malloc(tile_size);
  if (TILEbuf == (unsigned char *) NULL)
  {
    fprintf(stderr,
	    "Error: unable to allocate space for tile buffer!\n");
    fclose(DDBfp);
    fclose(RAWfp);
    unlink(RAWname);
    free(MAINbuf);
    exit(1);
  }
  /*******************************************************************
   *                        PROCESSING LOOP                          *
   *******************************************************************
   * For each tile:                                                  *
   * -------------                                                   *
   *                                                                 *
   * NOTE: If DATAval==ORIGTYPE, just write out tile...              *
   *                                                                 *
   *  a) Get TILE of data from file..                                *
   *  b) Re-cast TILE buffer as COMPLEX...                           *
   *  c) Process each tile. For each complex pixel in tile buffer    *
   *     -- If output datatype = original, write to output file..    *
   *     -- If output datatype = mag32, convert to 32-bit float mag, *
   *        and write to output file..                               *
   *     -- If output datatype = mag32, convert to 32-bit float mag, *
   *        update max and min values, store in main (float) buffer. *
   *        Later, we will scale main float buffer to 8-bit and write*
   *        whole 8-bit buffer to output file.                       *
   *  d) Update percent read variables...let user know...            *
   *******************************************************************/

  /* Set up MAX & MIN variable, if needed */
  if (DATAval == MAG8TYPE)
  {
    fmax = -100000.0;
    fmin = 100000.0;
  }
  /* Determine TILE start & end */
  switch (TILEflag)
  {
  case FALSE:			/* Process ALL data tiles */
    TILEstart = 0;
    TILEend = numtiles;

    /* Point to beginning of DDB complex data.. */
    magloc = (long) phlen;
    fseek(DDBfp, magloc, 0);

    break;

  case TRUE:			/* Process 1 tile */
    TILEstart = TILEval;
    TILEend = TILEval + 1;

    /* Point to beginning of DDB complex TILE data.. */
    magloc = (long) phlen + (TILEstart * tile_size);
    fseek(DDBfp, magloc, 0);

    break;
  }

  /* Process FULL tiles first */
  printf("Performing Initial Processing:\n\n");

  for (i = TILEstart; i < TILEend; i++)
  {
    /* Get 1 Full TILE of data */
    numgot = fread(TILEbuf, sizeof(char), tile_size, DDBfp);

    if (DATAval == ORIGTYPE)
    {
      numwrote = fwrite(TILEbuf, sizeof(char), tile_size, RAWfp);
    }
    /* Re-cast as COMPLEX */
    C_TILEbuf = (struct cmplxnum *) TILEbuf;

    /* Process each TILE */
    for (j = 0; j < numFullTilePixels; j++)
    {

      switch (DATAval)
      {
      case MAG8TYPE:
      case MAG32TYPE:
	/* Convert pixel to float magnitude value */
	dreal = (double) (C_TILEbuf[j].r * C_TILEbuf[j].r);
	dimag = (double) (C_TILEbuf[j].i * C_TILEbuf[j].i);
	d1 = sqrt(dreal + dimag);
	magval = (float) d1;

	/* Update float max and min values, if MAG8TYPE */
	if (DATAval == MAG8TYPE)
	{
	  fmax = MAX(fmax, magval);
	  fmin = MIN(fmin, magval);
	}
	/* Store in main buffer */
	switch (TILEflag)
	{
	case FALSE:		/* Do ALL data */
	  MAINbuf[j + (i * numFullTilePixels)] = magval;
	  break;
	case TRUE:		/* Do 1 TILE   */
	  MAINbuf[j] = magval;
	  break;
	}
	break;
      }				/* Switch end */

    }

    /* Calculate percent complete */
    if (TILEflag == FALSE)
    {
      percentComplete = percentEachRead + (i * percentEachRead);
      printf("  [%6.3f] percent completed...\n", percentComplete);
    }
  }

  /* Process (last) PARTIAL Tile (if TILEflag = FALSE & there is data) */

  if ((TILEflag == FALSE) &&
      (partialTileSize != 0))
  {
    numgot = fread(TILEbuf, sizeof(char), partialTileSize, DDBfp);
    if (numgot != partialTileSize)
    {
      fprintf(stderr, "Error: in reading remaining JumpStart DDB data!\n");
      fprintf(stderr, "Only read [%d] of [%d] pixels!\n\n",
	      numwrote, partialTileSize);
      fclose(DDBfp);
      fclose(RAWfp);
      free(MAINbuf);
      free(TILEbuf);
      C_TILEbuf = NULL;
      exit(1);
    }
    if (DATAval == ORIGTYPE)
    {
      numwrote = fwrite(TILEbuf, sizeof(char), partialTileSize, RAWfp);
    }
    /* Re-cast as COMPLEX */
    C_TILEbuf = (struct cmplxnum *) TILEbuf;

    /* Convert partial tile data */
    for (j = 0; j < numPartTilePixels; j++)
    {

      switch (DATAval)
      {
      case MAG8TYPE:
      case MAG32TYPE:
	/* Convert pixel to float magnitude value */
	dreal = (double) (C_TILEbuf[j].r * C_TILEbuf[j].r);
	dimag = (double) (C_TILEbuf[j].i * C_TILEbuf[j].i);
	d1 = sqrt(dreal + dimag);
	magval = (float) d1;

	/* Update float max and min values (if MAG8TYPE) */
	if (DATAval == MAG8TYPE)
	{
	  fmax = MAX(fmax, magval);
	  fmin = MIN(fmin, magval);
	}
	/* Store in main buffer */
	MAINbuf[j + (numFullTilePixels * numtiles)] = magval;
	break;
      }

    }

  }
  /* Close input file..free memory..NULL pointers */
  fclose(DDBfp);
  free(TILEbuf);
  C_TILEbuf = NULL;

  printf("\nInitial processing complete...\n");


  /******************* FINAL PROCESSING AREA ***********************/
  switch (DATAval)
  {
  case ORIGTYPE:		/* COMPLEX */
    fclose(RAWfp);
    break;
  case MAG32TYPE:		/* 32-bit FLOAT */
    printf("\nWriting 32-bit float JumpStart DDB data to [%s]\n", RAWname);
    numwrote = fwrite(MAINbuf, sizeof(float), npix, RAWfp);
    if (numwrote != npix)
    {
      fprintf(stderr, "Error: in writing out 32-bit float JumpStart DDB data!\n");
      fprintf(stderr, "Only wrote [%d] of [%d] pixels!\n\n", numwrote, npix);
      free(MAINbuf);
      fclose(DDBfp);
      fclose(RAWfp);
      unlink(RAWname);
      exit(1);
    }
    free(MAINbuf);
    fclose(RAWfp);
    break;
  case MAG8TYPE:		/* 8-BIT */
    printf("Scaling JumpStart DDB image data to 8-bit...\n");

    /* Allocate 8-bit buffer */
    pic8 = (unsigned char *) calloc((size_t) npix, (size_t) 1);
    if (pic8 == (unsigned char *) NULL)
    {
      fprintf(stderr,
	      "Error: Unable to malloc output 8-bit memory!\n");
      free(MAINbuf);
      fclose(RAWfp);
      unlink(RAWname);
      exit(1);
    }
    /* Do 8-bit linear scaling of MAINbuf data */
    frange = fmax - fmin;
    fscale = 255.0 / frange;

    for (i = 0; i < npix; i++)
    {
      f1 = (float) MAINbuf[i] - fmin;
      f2 = f1 * fscale;
      pic8[i] = (unsigned char) (f2 + 0.5);
    }

    free(MAINbuf);

    /* Write 8-bit data out to output file */
    printf("\nWriting 8-bit linearly-scaled JumpStart DDB data to [%s]\n", RAWname);
    numwrote = fwrite(pic8, sizeof(char), npix, RAWfp);
    if (numwrote != npix)
    {
      fprintf(stderr, "Error: in writing out 8-bit JumpStart DDB data!\n");
      fprintf(stderr, "Only wrote [%d] of [%d] pixels!\n\n", numwrote, npix);
      fclose(DDBfp);
      fclose(RAWfp);
      unlink(RAWname);
      exit(1);
    }
    free(pic8);
    fclose(RAWfp);
    break;
  }


  printf("\nJumpStart DDB to RAW Binary conversion: completed!\n\n");

  exit(0);
}
