
/*****************************************************************************
 * This software and any accompanying documentation is released "as is."  The 
 * government makes no warranty of any kind, express or implied, 
 * concerning this software and any accompanying documentation, including, 
 * without limitation, any warranties of merchantability or fitness for a 
 * particular purpose.  In no event will the U.S. government be liable for any
 * damages, including any lost profits, lost savings or other incidental or 
 * consequential damages arising out of the use, or inability to use, this 
 * software or any accompanying documentation, even if informed in advance of 
 * the possibility of such damages.                
 /****************************************************************************/

/*****************************************************************************

 * Subroutines Included:  
 *   get_hdr_value
 *      search_hdr_parameter
 *      get_header_names
 *      put_hdr_value
 *      remove_hdr_parameter
 *      get_mem_hdr_value
 *      fill_hdr_parameters
 *      free_hdr_parameters
 *
 * Revision History:
 *       2/2/98 - Created get_hdr_value - DH
 *       2/2/98 - Created search_hdr_paramerets - DH
 *       2/2/98 - Created get_header_names - RTB
 *       2/2/98 - Created put_hdr_values - DH
 *       2/2/98 - Created remove_hdr_parameter - DH
 *       2/2/98 - Created get_mem_hdr_value - DH
 *       2/2/98 - Created fill_hdr_parameters - DH
 *       2/2/98 - Created free_hdr_parameters - DH
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <malloc.h>
#include "phoenix.h"
#include "private.h"


extern hdr_parameters *phoenix_parameters;
extern hdr_parameters *standard_parameters;

/**************************************************************************

 * get_hdr_value: get the value for a specific parameter in the Phoenix header
 *
 * Parameters: parameter_name - string to identify the parameter to be retrieved
 *             value - value of the header parameter
 *             return_type - the type to return the header parameter 
 *
 * Return: 0 - not found
 *         1 - found
 *
 *************************************************************************/

/**************************************************************************
 ***    When calling this routine the type of the value argument can either
 ***    be a phoenix_type pointer or it can be a return_type pointer
 **************************************************************************/

int
get_hdr_value (parameter_name, value, return_type)
     char parameter_name[MAX_STRING_LEN];
     phoenix_type *value;
     int return_type;
{
  char temp[MAX_STRING_LEN];
  int found = FALSE;
  float temp_float;
  int temp_int;
  double temp_double;
  short temp_short;

  found = search_hdr_parameter (parameter_name, temp);
  if (!found)
    {
      printf ("'%s' is not a valid parameter name\n", parameter_name);
    }
  else
    {
      switch (return_type)
	{
	case FLOAT:
	  temp_float = (float) atof (temp);
	  value->float_value = temp_float;
	  break;
	case INT:
	  temp_int = (int) atoi (temp);
	  value->int_value = temp_int;
	  break;
	case DOUBLE:
	  temp_double = (double) atof (temp);
	  value->double_value = temp_double;
	  break;
	case ASCII:
	  strcpy (value->string_value, temp);
	  break;
	case SHORT:
	  temp_short = (short) atoi (temp);
	  value->short_value = temp_short;
	  break;
	}
    }


  return (!found);
}

/**************************************************************************

 * search_hdr_parameter: get the value for a specific parameter in the  
 *                       Phoenix header
 *
 * Parameters: parameter - string to identify the parameter to be retrieved
 *             value - string containing the parameter retrieved
 *
 * Return: 0 - not found
 *         1 - found
 *
 **************************************************************************/
int
search_hdr_parameter (parameter, value)
     char *parameter;
     char *value;
{
  int found;
  hdr_parameters *ptr;

  ptr = phoenix_parameters;
  found = FALSE;
  while ((ptr) && (!found))
    {
      if (!strcmp (parameter, ptr->name))
	{
	  strcpy (value, ptr->value);
	  found = TRUE;
	}
      else
	{
	  ptr = ptr->next;
	}
    }
  return found;
}


/**************************************************************************

 * get_header_names: get a list of all Phoenix parameters containing a given string
 *
 * Parameters: search - string to identify the parameters to be retrieved
 *             array - list of all qualifing names
 *             max - maximum number of names
 *
 * Return: number of names found
 *
 **************************************************************************/
int
get_header_names (search, array, max)
     char *search;
     char *array[];
     int max;
{
  int count = 0;
  hdr_parameters *ptr;
  char *parameter, *look;
  int len, i;

  len = strlen (search);
  look = (char *) malloc (len + 1);
  for (i = 0; i < len; i++)
    {
      if ((search[i] >= 'A') && (search[i] <= 'Z'))
	look[i] = search[i] - 'A' + 'a';
      else
	look[i] = search[i];
    }
  look[len] = '\0';

  ptr = phoenix_parameters;
  while (ptr)
    {
      len = strlen (ptr->name);
      parameter = (char *) malloc (len + 1);
      for (i = 0; i < len; i++)
	{
	  if ((ptr->name[i] >= 'A') && (ptr->name[i] <= 'Z'))
	    parameter[i] = ptr->name[i] - 'A' + 'a';
	  else
	    parameter[i] = ptr->name[i];
	}
      parameter[len] = '\0';
      if (strstr (parameter, look) != NULL)
	{
	  if (count < max)
	    {
	      array[count] = (char *) malloc (len + 1);
	      strcpy (array[count], ptr->name);
	    }
	  count++;
	}
      ptr = ptr->next;
      free (parameter);
    }
  return count;
}

/**************************************************************************

 * put_hdr_value: Add or update a value to the Phoenix Header
 *
 * Parameters: parameter_name - string to identify the parameter
 *             value - value of the header parameter
 *             input - type of the header parameter
 *
 * Return: 0 - Parameter added to Phoenix header
 *         1 - Parameter updated in Phoenix Header
 *
 **************************************************************************/

/**************************************************************************
 ***    When calling this routine the type of the value argument can either
 ***    be a phoenix_type variable or it can be a return_type pointer
 **************************************************************************/

int
put_hdr_value (parameter_name, value, input)
     char parameter_name[MAX_STRING_LEN];
     phoenix_type value;
     int input;
{
  int found;
  int i = 0;
  hdr_parameters *ptr, *prev;
  char temp[MAX_STRING_LEN];

  ptr = phoenix_parameters;
  found = FALSE;
  while ((ptr) && (!found))
    {
      if (!strcmp (parameter_name, ptr->name))
	{
	  switch (input)
	    {
	    case FLOAT:
	      sprintf (ptr->value, "%f", value.float_value);
	      break;
	    case INT:
	      sprintf (ptr->value, "%d", value.int_value);
	      break;
	    case DOUBLE:
	      sprintf (ptr->value, "%lf", value.double_value);
	      break;
	    case ASCII:
	      free (ptr->value);
	      ptr->value = (char *) malloc (strlen (value.string_value) + 1);
	      sprintf (ptr->value, "%s", value.string_value);
	      break;
	    case SHORT:
	      sprintf (ptr->value, "%hd", value.short_value);
	      break;
	    }
	  found = TRUE;
	}
      else
	{
	  prev = ptr;
	  ptr = ptr->next;
	}
    }

  if (!found)
    {
      ptr = (hdr_parameters *) malloc (sizeof (hdr_parameters));
      ptr->name = (char *) malloc (strlen (parameter_name));
      strcpy (ptr->name, parameter_name);

      switch (input)
	{
	case FLOAT:
	  sprintf (temp, "%f", value.float_value);
	  break;
	case INT:
	  sprintf (temp, "%d", value.int_value);
	  break;
	case DOUBLE:
	  sprintf (temp, "%lf", value.double_value);
	  break;
	case ASCII:
	  sprintf (temp, "%s", value.string_value);
	  break;
	case SHORT:
	  sprintf (temp, "%hd", value.short_value);
	  break;
	}
      ptr->value = (char *) malloc (strlen (temp));
      strcpy (ptr->value, temp);
      ptr->next = NULL;
      prev->next = ptr;

    }
  return found;
}

/**************************************************************************

 * remove_hdr_parameter: Remove a value from the Phoenix header
 *
 * Parameters: parameter - string to identify the parameter to be removed
 *
 * Return: 0 - value not in phoenix header
 *         1 - value removed from phoenix header
 *
 **************************************************************************/
int
remove_hdr_parameter (parameter)
     char *parameter;
{
  int found;
  hdr_parameters *ptr, *prev;

  ptr = phoenix_parameters;
  prev = NULL;
  found = FALSE;
  while ((ptr) && (!found))
    {
      if (!strcmp (parameter, ptr->name))
	{
	  if (prev)
	    {
	      prev->next = ptr->next;
	      free (ptr);
	    }
	  else
	    {
	      phoenix_parameters = ptr->next;
	      free (ptr);
	    }
	  found = TRUE;
	}
      else
	{
	  prev = ptr;
	  ptr = ptr->next;
	}
    }
  if (!found)
    {
      printf ("'%s' is not a valid parameter name\n", parameter);
    }
  return found;
}

/**************************************************************************

 * get_mem_hdr_value: Search the signature data for a specifed parameter
 *
 * Parameters: parameter_name - string to identify the parameter to be retrieved
 *             value - string containing the value of the specified parameter
 *
 * Return: 0 - parameter found
 *         1 - parameter not found
 *
 **************************************************************************/
int
get_mem_hdr_value (parameter_name, value)
     char parameter_name[MAX_STRING_LEN];
     char *value;
{
  char *sig_buf;
  char *tmp;
  char line[MAX_STRING_LEN];
  int done = FALSE;


  tmp = value;
  sig_buf = (char *) get_status (LOCATION);
  while (!done && (sig_buf[0] != '\0'))
    {
      if (!strncmp (sig_buf, parameter_name, strlen (parameter_name)))
	{
	  while (sig_buf[0] != '=')
	    sig_buf++;
	  sig_buf++;

	  while (sig_buf[0] != '\n')
	    {
	      value++[0] = sig_buf++[0];
	    }
	  value[0] = '\0';
	  done = TRUE;

	}
      else
	sig_buf++;
    }

  return (!done);
}

/**************************************************************************

 * fill_hdr_parameters: Initalize the global Parameter with the currently
 *                      loaded Signature data.
 *
 **************************************************************************/
int
fill_hdr_parameters ()
{
  char *sig_ptr, *start;
  char string_temp[MAX_STRING_LEN];
  unsigned long header_size, length, native_start;
  short native_set = FALSE;
  char parameter_name[MAX_STRING_LEN], *f;
  char value[MAX_STRING_LEN], *v;
  hdr_parameters *temp, *ptr;
  int x;

  sig_ptr = (char *) get_status (LOCATION);

  /*  skip initial newline  */
  while(sig_ptr[0] != '\n') sig_ptr++;
  sig_ptr++;

  ptr = standard_parameters;
  for (x = 0; x < 3; x++)
    {
      f = parameter_name;
      while (sig_ptr[0] != '\n')
	f++[0] = sig_ptr++[0];
      sig_ptr++;
      f[0] = '\0';

      if (!strncmp (parameter_name, "PhoenixHeaderLength", 19))
	{
	  sscanf (parameter_name, "%*s %ld", &header_size);
	  set_status (PHOENIX_SIZE, header_size);
	  start = (char *) get_status (LOCATION);
	  set_status (NATIVE_HDR_ADDR,
		      (unsigned long) (start + header_size));
	}

/*
      if (!strncmp (parameter_name, "PhoenixSigSize", 14))
	{
	  sscanf (parameter_name, "%*s %ld", &header_size);
	  set_status (RECORD_SIZE, header_size);
	}
*/
      temp = (hdr_parameters *) malloc (sizeof (hdr_parameters));
      temp->name = (char *) malloc (strlen (parameter_name));
      strcpy (temp->name, parameter_name);
      temp->value = NULL;
      temp->next = NULL;
      if (!ptr)
	{
	  standard_parameters = temp;
	  ptr = standard_parameters;
	}
      else
	{
	  ptr->next = temp;
	  ptr = temp;
	}
    }

  ptr = phoenix_parameters;
  while (strncmp (sig_ptr, "[EndofPhoenixHeader]", 20))
    {
      /*  get parameter name  */
      f = parameter_name;
      while (sig_ptr[0] != '=')
	f++[0] = sig_ptr++[0];
      sig_ptr++;
      f[0] = '\0';

      /* skip over blanks after = */
      while (sig_ptr[0] == ' ')
	sig_ptr++;

      /*  get value  */
      v = value;
      while (sig_ptr[0] != '\n')
	v++[0] = sig_ptr++[0];
      v[0] = '\0';
      sig_ptr++;

      if (!strcmp (parameter_name, "native_header_length"))
	{
	  length = (unsigned long) atol (value);
	  set_status (NATIVE_HDR_SIZE, length);
	  native_start = get_status (NATIVE_HDR_ADDR);
	  set_status (DATA_ADDR, native_start + length);
	  native_set = TRUE;
	}
      temp = (hdr_parameters *) malloc (sizeof (hdr_parameters));
      temp->name = (char *) malloc (strlen (parameter_name));
      temp->value = (char *) malloc (strlen (value));
      strcpy (temp->name, parameter_name);
      strcpy (temp->value, value);
      temp->next = NULL;
      if (!ptr)
	{
	  phoenix_parameters = temp;
	  ptr = phoenix_parameters;
	}
      else
	{
	  ptr->next = temp;
	  ptr = temp;
	}

    }

  if (!native_set)
    {
      fprintf (stderr, "Warning: fill_hdr_parameters: No native Header length\n");
      set_status (NATIVE_HDR_SIZE, 0);
    }


}

/**************************************************************************

 * free_hdr_parameters: Empty the current set of Global header Parameters
 *
 **************************************************************************/
void
free_hdr_parameters ()
{
  hdr_parameters *ptr, *temp;

  ptr = standard_parameters;
  while (ptr)
    {
      temp = ptr;
      ptr = ptr->next;
      free (temp->name);
      free (temp);
    }
  standard_parameters = NULL;

  ptr = phoenix_parameters;
  while (ptr)
    {
      temp = ptr;
      ptr = ptr->next;
      free (temp->name);
      free (temp->value);
      free (temp);
    }
  phoenix_parameters = NULL;
}
