/*****************************************************************************
 * This software and any accompanying documentation is released "as is."  The 
 * government makes no warranty of any kind, express or implied, 
 * concerning this software and any accompanying documentation, including, 
 * without limitation, any warranties of merchantability or fitness for a 
 * particular purpose.  In no event will the U.S. government be liable for any
 * damages, including any lost profits, lost savings or other incidental or 
 * consequential damages arising out of the use, or inability to use, this 
 * software or any accompanying documentation, even if informed in advance of 
 * the possibility of such damages.                
 /****************************************************************************/

/*****************************************************************************

 * Subroutines Included:  
 *       read_sig
 *      read_hdr
 *      get_data_size
 *      get_hdr_size
 *      buffered_read
 *
 * Revision History:
 *       2/2/98 - Created read_sig - DH & RTB
 *       2/2/98 - Created read_hdr - DH & RTB
 *       2/2/98 - Created get_data_size - DH
 *       2/2/98 - Created get_hdr_size - DH
 *       2/2/98 - Created buffered_read - DH & RTB
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include "phoenix.h"
#include "private.h"

/**************************************************************************

 * read_sig: Read the Phoenix Header and data of a file
 *
 * Parameters: fp - File pointer of the file to be read
 *
 * return: Error Code
 *
 **************************************************************************/
int
read_sig (fp)
     FILE *fp;
{
  static void *sig_buf = NULL;
  static unsigned long available = 0;
  int results, errcode;
  unsigned long data_size;
  char *s, parameter_name[MAX_STRING_LEN];
  unsigned long bytes_remaining, bytes_to_copy;


  errcode = 0;

  if (sig_buf == NULL)
    {
      sig_buf = (void *) malloc (TAPE_BLOCK_SIZE);
      available = TAPE_BLOCK_SIZE;
      set_status (LOCATION, (unsigned long) sig_buf);
    }
  if (sig_buf == NULL)
    errcode = ALLOC_ER;

  if (!errcode)
    {
      results = buffered_read (fp, (char *) sig_buf, BUFFER_LEN);
      if (results == 0)
	errcode = EOF_ER;
      else if (results == -1)
	errcode = FREAD_ERR;
      else if (results < BUFFER_LEN)
	errcode = EOF_ER;
    }

  if (!errcode)
    data_size = get_data_size ();

  if (errcode)
    return errcode;

  if (data_size > available)
    {
      sig_buf = (void *) realloc ((char *) sig_buf, data_size);
      available = data_size;

      if (sig_buf == NULL)
	errcode = ALLOC_ER;
      else
	set_status (LOCATION, (unsigned long) sig_buf);
    }

  if (!errcode)
    {
      s = (char *) sig_buf;
      s += BUFFER_LEN;
      bytes_to_copy = data_size - BUFFER_LEN;

      results = buffered_read (fp, s, bytes_to_copy);

      if (results == 0)
	errcode = EOF_ER;
      else if (results == -1)
	errcode = FREAD_ERR;
      else if (results < bytes_to_copy)
	errcode = EOF_ER;

    }

  results = buffered_read (fp, "Flush", 0);

  if (!errcode)
    {
      free_hdr_parameters ();
      fill_hdr_parameters ();
    }

  return errcode;
}

/**************************************************************************

 * read_hdr: Read the Phoenix Header of a file
 *
 * Parameters: fp - File pointer of the file to be read
 *
 * return: Error Code
 *
 **************************************************************************/
int
read_hdr (fp)
     FILE *fp;
{
  static void *sig_buf = NULL;
  static unsigned long available = 0;
  int results, errcode;
  unsigned long hdr_size;
  char *s, parameter_name[MAX_STRING_LEN];
  unsigned long bytes_remaining, bytes_to_copy;


  errcode = 0;

  if (sig_buf == NULL)
    {
      sig_buf = (void *) malloc (TAPE_BLOCK_SIZE);
      available = TAPE_BLOCK_SIZE;
      set_status (LOCATION, (unsigned long) sig_buf);
    }
  if (sig_buf == NULL)
    errcode = ALLOC_ER;

  if (!errcode)
    {
      results = buffered_read (fp, (char *) sig_buf, BUFFER_LEN);
      if (results == 0)
	errcode = EOF_ER;
      else if (results == -1)
	errcode = FREAD_ERR;
      else if (results < BUFFER_LEN)
	errcode = EOF_ER;
    }

  if (!errcode)
    hdr_size = get_hdr_size ();

  if (errcode)
    return errcode;

  if (hdr_size > available)
    {
      sig_buf = (void *) realloc ((char *) sig_buf, hdr_size);
      available = hdr_size;

      if (sig_buf == NULL)
	errcode = ALLOC_ER;
      else
	set_status (LOCATION, (unsigned long) sig_buf);
    }

  if (!errcode)
    {
      s = (char *) sig_buf;
      s += BUFFER_LEN;
      bytes_to_copy = hdr_size - BUFFER_LEN;

      results = buffered_read (fp, s, bytes_to_copy);

      if (results == 0)
	errcode = EOF_ER;
      else if (results == -1)
	errcode = FREAD_ERR;
      else if (results < bytes_to_copy)
	errcode = EOF_ER;

    }

  results = buffered_read (fp, "Flush", 0);

  if (results == -1)
    errcode = FREAD_ERR;

  if (!errcode)
    {
      free_hdr_parameters ();
      fill_hdr_parameters ();
    }

  return errcode;
}

/**************************************************************************

 * get_data_size: get the size of the data file from the header
 *
 * return: Size in bytes of the file
 *
 **************************************************************************/
unsigned long
get_data_size ()
{
  char *sig_buf;
  char temp[MAX_STRING_LEN];
  unsigned long value;
  int errcode;



  errcode = get_mem_hdr_value ("PhoenixSigSize", temp);

  if (errcode)
    {
      process_err ("get_data_size", NO_SIG_ERR);
      iQuit (1);
    }
  sscanf (temp, "%ld", &value);
  set_status (RECORD_SIZE, value);

  if (value < BUFFER_LEN)
    {
      process_err ("get_data_size", BUFFER_TO_BIG);
      iQuit (1);
    }
  return value;
}


/**************************************************************************

 * get_hdr_size: get the size of the phoenix header
 *
 * return: Size in bytes of the header
 *
 **************************************************************************/
unsigned long
get_hdr_size ()
{
  char *sig_buf;
  char temp[MAX_STRING_LEN];
  unsigned long value;
  int errcode;



  errcode = get_mem_hdr_value ("PhoenixSigSize", temp);

  if (errcode)
    {
      process_err ("get_hdr_size", NO_SIG_ERR);
      iQuit (1);
    }
  sscanf (temp, "%ld", &value);
  set_status (RECORD_SIZE, value);

  if (value < BUFFER_LEN)
    {
      process_err ("get_hdr_size", BUFFER_TO_BIG);
      iQuit (1);
    }


  errcode = get_mem_hdr_value ("PhoenixHeaderLength", temp);
  sscanf (temp, "%ld", &value);

  return value;
}






/**************************************************************************

 * buffered_read: Read data from a file
 *
 * Parameters: fh - file handler
 *             ptr - string of read data
 *             b - number of bytes to read
 *
 * return: bytes read
 *
 **************************************************************************/
static int
buffered_read (fh, ptr, b)
     FILE *fh;
     char *ptr;
     int b;
{
  static char block_buffer[TAPE_BLOCK_SIZE];
  int bytes_to_copy;
  int rc;
  char *s;
  int bytes_loaded;
  static int data_start;

  if (!get_status (BUFFERED_READ))
    return fread (ptr, sizeof (char), b, fh);

  if (strcmp (ptr, "Flush") == 0)
    {
      data_start = 0;
      set_status (BLOCK_LOADED, FALSE);
      rc = fread (block_buffer, sizeof (char), TAPE_BLOCK_SIZE, fh);
    }

  bytes_loaded = 0;

  while (b > 0)
    {
      if (!get_status (BLOCK_LOADED))
	{
	  rc = fread (block_buffer, sizeof (char), TAPE_BLOCK_SIZE, fh);

	  if (rc < 0)
	    return rc;
	  set_status (BLOCK_LOADED, TRUE);
	  data_start = 0;
	}
      if (TAPE_BLOCK_SIZE - data_start > b)
	bytes_to_copy = b;
      else
	bytes_to_copy = TAPE_BLOCK_SIZE - data_start;

      memcpy (ptr, &block_buffer[data_start], bytes_to_copy);
      b -= bytes_to_copy;
      ptr += bytes_to_copy;
      data_start += bytes_to_copy;
      bytes_loaded += bytes_to_copy;
      if (data_start >= TAPE_BLOCK_SIZE)
	set_status (BLOCK_LOADED, FALSE);
    }
  return bytes_loaded;
}
