/*****************************************************************************
 * This software and any accompanying documentation is released "as is."  The 
 * government makes no warranty of any kind, express or implied, 
 * concerning this software and any accompanying documentation, including, 
 * without limitation, any warranties of merchantability or fitness for a 
 * particular purpose.  In no event will the U.S. government be liable for any
 * damages, including any lost profits, lost savings or other incidental or 
 * consequential damages arising out of the use, or inability to use, this 
 * software or any accompanying documentation, even if informed in advance of 
 * the possibility of such damages.                
 /****************************************************************************/

/*****************************************************************************
 * Subroutines Included:  
 *      write_sig
 *      write_hdr
 *      calc_phoenix_hdr_length
 *      buffered_write
 *
 * Revision History:
 *       2/2/98 - Created write_sig - DH & RTB
 *       2/2/98 - Created write_hdr - DH & RTB
 *       2/2/98 - Created calc_phoenix_hdr_length - DH
 *       2/2/98 - Created buffered_write - DH & RTB
 *       10/19/98 - Made Phoenix Sig Size contain 10 characters - RTB
 *       03/17/99 - Move Phoenix Line to front of list automatically - RTB
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include "phoenix.h"
#include "private.h"

extern hdr_parameters *phoenix_parameters;
extern hdr_parameters *standard_parameters;

/**************************************************************************

 * write_sig: write the data to a file
 *
 * Parameters: fp - File pointer of the file to be written
 *             sig_addr - address of the signature data
 *             sig_size - number of bytes to be written
 *
 * return: Error Code
 *
 **************************************************************************/
int
write_sig (fp, sig_addr, sig_size)
     FILE *fp;
     void *sig_addr;
     int sig_size;
{
  char	dummy[15];

  typedef struct
    {
      char a[TAPE_BLOCK_SIZE];
    }
  dblock;

  int results, errcode, temp;
  unsigned long bytes_to_write;
  dblock last_block;


  errcode = 0;

  if (!sig_addr)
    errcode = SIG_ADDRESS_ERR;
  if (!sig_size)
    errcode = SIG_LENGTH_ERR;

  if (errcode)
    return errcode;

  results = buffered_write (fp, (char *) sig_addr, sig_size);

  if (results < 0)
    errcode = FWRITE_ERR;
  else if (results < sig_size)
    errcode = DATA_WRITE_ERR;

  sprintf (dummy, "Flush", 0);
  results = buffered_write (fp, dummy, 0);

  if (results < 0)
    errcode = FWRITE_ERR;
  else if (results != 0)
    errcode = DATA_WRITE_ERR;

  return errcode;
}

/**************************************************************************

 * write_hdr: write the Phoenix header to a file
 *
 * Parameters: fp - File pointer of the file to be written
 *
 * return: Error Code
 *
 **************************************************************************/
/***************************************************************************
 * KNOWN BUG:  When writting in buffer mode the last partial block of the header
 *             is not written.  This is necessary to write both a hdr and 
 *             data to a tape file.  If just a header is desired after the
 *             write_hdr call buffered_write(fp,"Flush",0);
 ***************************************************************************/
int
write_hdr (fp)
     FILE *fp;
{
  hdr_parameters *ptr,*head,*prev;
  char str[MAX_STRING_LEN];
  int results, errcode = 0;
  char temp_char[10];


  strcpy (temp_char, "XXXX");
  put_hdr_value ("PhoenixLines", temp_char, ASCII);
  calc_phoenix_hdr_length ();

  sprintf (str, "\n");
  results = buffered_write (fp, str, strlen (str));

  if (results < 0)
    errcode = FWRITE_ERR;
  else if (results < strlen (str))
    errcode = DATA_WRITE_ERR;

  if (errcode)
    return errcode;

  ptr = standard_parameters;

  while (ptr)
    {
      sprintf (str, "%s\n", ptr->name);
      results = buffered_write (fp, str, strlen (str));
      if (results != strlen (str))
	break;
      ptr = ptr->next;
    }

  if (results < 0)
    errcode = FWRITE_ERR;
  else if (results < strlen (str))
    errcode = DATA_WRITE_ERR;

  if (errcode)
    return errcode;

  ptr = phoenix_parameters;
  head = phoenix_parameters;
  prev = NULL;
  while (ptr)
    {
      if(!strcmp(ptr->name,"PhoenixLines")) {
        if(prev == NULL) break;
        prev->next = ptr->next;
        ptr->next = head;
        phoenix_parameters = ptr;
        break;
      }
      prev = ptr;
      ptr = ptr->next;
    }

  ptr = phoenix_parameters;
  while (ptr)
    {
      sprintf (str, "%s= %s\n", ptr->name, ptr->value);
      results = buffered_write (fp, str, strlen (str));
      if (results != strlen (str))
	break;
      ptr = ptr->next;
    }

  if (results < 0)
    errcode = FWRITE_ERR;
  else if (results < strlen (str))
    errcode = DATA_WRITE_ERR;

  if (errcode)
    return errcode;

  sprintf (str, "%s\n", "[EndofPhoenixHeader]");
  results = buffered_write (fp, str, strlen (str));

  if (results < 0)
    errcode = FWRITE_ERR;
  else if (results < strlen (str))
    errcode = DATA_WRITE_ERR;

  if (results < 0)
    errcode = FWRITE_ERR;
  else if (results != 0)
    errcode = DATA_WRITE_ERR;

  return errcode;
}

/**************************************************************************

 * calc_phoenix_hdr_length: Check and Adjust the length of the Phoenix header
 *
 **************************************************************************/
int
calc_phoenix_hdr_length ()
{
  hdr_parameters *ptr;
  unsigned long standard_len = 0, phoenix_len = 0;
  unsigned long end_len = 0, new_phoenix_len, initial_phoenix_len;
  unsigned long initial_total_len, new_total_len, remainder;
  char str[MAX_STRING_LEN], temp_char[10];
  int total_lines;


  total_lines = 1;
  ptr = standard_parameters;
  while (ptr)
    {
      standard_len += strlen (ptr->name) + 1;
      if((strncmp(ptr->name,"PhoenixSigSize=",15) == 0) &&
          strlen(ptr->name) != 26) {
         standard_len += 26 - strlen(ptr->name);
      }
/*
      printf("%s\n",ptr->name);
*/
      ptr = ptr->next;
      total_lines++;
    }

  ptr = phoenix_parameters;
  while (ptr)
    {
      phoenix_len += strlen (ptr->name) + strlen (ptr->value) + 3;
      ptr = ptr->next;
      total_lines++;
    }

  end_len = strlen ("[EndofPhoenixHeader]\n");
  total_lines++;

  new_phoenix_len = standard_len + phoenix_len + end_len + 1;

  ptr = standard_parameters;
  ptr = ptr->next;
  sscanf (ptr->name, "%*s %ld", &initial_phoenix_len);
  sprintf (str, "PhoenixHeaderLength= %05ld", new_phoenix_len);
  free (ptr->name);
  ptr->name = (char *) malloc (strlen (str));
  if (ptr->name)
    {
      strcpy (ptr->name, str);
      set_status (PHOENIX_SIZE, new_phoenix_len);
    }
  else
    {
      process_err ("calc_phoenix_hdr_length", ALLOC_ER);
      iQuit (1);
    }

  ptr = ptr->next;
  sscanf (ptr->name, "%*s %ld", &initial_total_len);
  new_total_len = initial_total_len - initial_phoenix_len
      + new_phoenix_len;

  sprintf (str, "PhoenixSigSize= %010ld", new_total_len);
  free (ptr->name);
  ptr->name = (char *) malloc (strlen (str));
  if (ptr->name)
    {
      strcpy (ptr->name, str);
      set_status (RECORD_SIZE, new_total_len);
    }
  else
    {
      process_err ("calc_phoenix_hdr_length", ALLOC_ER);
      iQuit (1);
    }

  sprintf (temp_char, "%04d", total_lines);
  put_hdr_value ("PhoenixLines", temp_char, ASCII);
}

	/*****************************************************
	 NAME: buffered_write
	 NOTES: like write, except it buffers writes into its
	 own local buffer, and always writes 8k blocks
	 at a time to the output file.
	 PARAMETERS:none
	 RETURNED VALUES:none
	 GLOBALS:none
	 PRIVATE:none
	 ERRORS:none
	 REVISION HISTORY: 
	 *****************************************************/
int
buffered_write (fh, ptr, b)
     FILE *fh;
     char *ptr;
     int b;
{
  static char block_buffer[8192];
  int bytes_to_copy;
  int rc;
  char *s;
  unsigned long data_start;
  short data_buffered;
  int bytes_copied = 0;
  int i;

  if (!get_status (BUFFERED_WRITE))
    return fwrite (ptr, sizeof (char), b, fh);

  data_start = get_status (WRITE_DATA_START);

  if (strcmp (ptr, "Flush") == 0)
    {
      rc = fwrite (block_buffer, sizeof (char), data_start, fh);

      if (rc != data_start)
	return FWRITE_ERR;
      data_buffered = FALSE;
      data_start = 0;
      bytes_copied = 0;
    }

  while (b > 0)
    {
      if (TAPE_BLOCK_SIZE - data_start > b)
	bytes_to_copy = b;
      else
	bytes_to_copy = TAPE_BLOCK_SIZE - data_start;

      memcpy (&block_buffer[data_start], ptr, bytes_to_copy);
      b -= bytes_to_copy;
      bytes_copied += bytes_to_copy;
      ptr += bytes_to_copy;
      data_start += bytes_to_copy;

      if (data_start >= TAPE_BLOCK_SIZE)
	{
	  rc = fwrite (block_buffer, sizeof (char), TAPE_BLOCK_SIZE, fh);

	  if (rc != TAPE_BLOCK_SIZE)
	    return FWRITE_ERR;
	  data_buffered = FALSE;
	  data_start = 0;
	}
      else
	{
	  if (data_start > 0)
	    data_buffered = TRUE;
	}
    }
  set_status (WRITE_DATA_START, data_start);
  set_status (DATA_BUFFERED, data_buffered);
  return bytes_copied;
}
